/*
 * Decompiled with CFR 0.152.
 */
package org.dspace.storage.bitstore;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.regions.Region;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.dspace.content.Bitstream;
import org.dspace.core.ConfigurationManager;
import org.dspace.core.Utils;
import org.dspace.storage.bitstore.BitStoreService;
import org.springframework.beans.factory.annotation.Required;

public class S3BitStoreService
implements BitStoreService {
    private static Logger log = Logger.getLogger(S3BitStoreService.class);
    private static final String CSA = "MD5";
    private String awsAccessKey;
    private String awsSecretKey;
    private String awsRegionName;
    private String bucketName = null;
    private String subfolder = null;
    private AmazonS3 s3Service = null;

    @Override
    public void init() throws IOException {
        if (StringUtils.isBlank((CharSequence)this.getAwsAccessKey()) || StringUtils.isBlank((CharSequence)this.getAwsSecretKey())) {
            log.warn((Object)"Empty S3 access or secret");
        }
        BasicAWSCredentials awsCredentials = new BasicAWSCredentials(this.getAwsAccessKey(), this.getAwsSecretKey());
        this.s3Service = new AmazonS3Client((AWSCredentials)awsCredentials);
        if (StringUtils.isEmpty((CharSequence)this.bucketName)) {
            this.bucketName = "dspace-asset-" + ConfigurationManager.getProperty("dspace.hostname");
            log.warn((Object)("S3 BucketName is not configured, setting default: " + this.bucketName));
        }
        try {
            if (!this.s3Service.doesBucketExist(this.bucketName)) {
                this.s3Service.createBucket(this.bucketName);
                log.info((Object)("Creating new S3 Bucket: " + this.bucketName));
            }
        }
        catch (Exception e) {
            log.error((Object)e);
            throw new IOException(e);
        }
        if (StringUtils.isNotBlank((CharSequence)this.awsRegionName)) {
            try {
                Regions regions = Regions.fromName((String)this.awsRegionName);
                Region region = Region.getRegion((Regions)regions);
                this.s3Service.setRegion(region);
                log.info((Object)("S3 Region set to: " + region.getName()));
            }
            catch (IllegalArgumentException e) {
                log.warn((Object)("Invalid aws_region: " + this.awsRegionName));
            }
        }
        log.info((Object)("AWS S3 Assetstore ready to go! bucket:" + this.bucketName));
    }

    @Override
    public String generateId() {
        return Utils.generateKey();
    }

    @Override
    public InputStream get(Bitstream bitstream) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        try {
            S3Object object = this.s3Service.getObject(new GetObjectRequest(this.bucketName, key));
            return object != null ? object.getObjectContent() : null;
        }
        catch (Exception e) {
            log.error((Object)("get(" + key + ")"), (Throwable)e);
            throw new IOException(e);
        }
    }

    @Override
    public void put(Bitstream bitstream, InputStream in) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        File scratchFile = File.createTempFile(bitstream.getInternalId(), "s3bs");
        try {
            FileUtils.copyInputStreamToFile((InputStream)in, (File)scratchFile);
            Long contentLength = scratchFile.length();
            PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucketName, key, scratchFile);
            PutObjectResult putObjectResult = this.s3Service.putObject(putObjectRequest);
            bitstream.setSizeBytes(contentLength);
            bitstream.setChecksum(putObjectResult.getETag());
            bitstream.setChecksumAlgorithm(CSA);
            scratchFile.delete();
        }
        catch (Exception e) {
            log.error((Object)("put(" + bitstream.getInternalId() + ", is)"), (Throwable)e);
            throw new IOException(e);
        }
        finally {
            if (scratchFile.exists()) {
                scratchFile.delete();
            }
        }
    }

    @Override
    public Map about(Bitstream bitstream, Map attrs) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        try {
            ObjectMetadata objectMetadata = this.s3Service.getObjectMetadata(this.bucketName, key);
            if (objectMetadata != null) {
                if (attrs.containsKey("size_bytes")) {
                    attrs.put("size_bytes", objectMetadata.getContentLength());
                }
                if (attrs.containsKey("checksum")) {
                    attrs.put("checksum", objectMetadata.getETag());
                    attrs.put("checksum_algorithm", CSA);
                }
                if (attrs.containsKey("modified")) {
                    attrs.put("modified", String.valueOf(objectMetadata.getLastModified().getTime()));
                }
                return attrs;
            }
        }
        catch (AmazonS3Exception e) {
            if (e.getStatusCode() == 404) {
                return null;
            }
        }
        catch (Exception e) {
            log.error((Object)("about(" + key + ", attrs)"), (Throwable)e);
            throw new IOException(e);
        }
        return null;
    }

    @Override
    public void remove(Bitstream bitstream) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        try {
            this.s3Service.deleteObject(this.bucketName, key);
        }
        catch (Exception e) {
            log.error((Object)("remove(" + key + ")"), (Throwable)e);
            throw new IOException(e);
        }
    }

    public String getFullKey(String id) {
        if (StringUtils.isNotEmpty((CharSequence)this.subfolder)) {
            return this.subfolder + "/" + id;
        }
        return id;
    }

    public String getAwsAccessKey() {
        return this.awsAccessKey;
    }

    @Required
    public void setAwsAccessKey(String awsAccessKey) {
        this.awsAccessKey = awsAccessKey;
    }

    public String getAwsSecretKey() {
        return this.awsSecretKey;
    }

    @Required
    public void setAwsSecretKey(String awsSecretKey) {
        this.awsSecretKey = awsSecretKey;
    }

    public String getAwsRegionName() {
        return this.awsRegionName;
    }

    public void setAwsRegionName(String awsRegionName) {
        this.awsRegionName = awsRegionName;
    }

    @Required
    public String getBucketName() {
        return this.bucketName;
    }

    public void setBucketName(String bucketName) {
        this.bucketName = bucketName;
    }

    public String getSubfolder() {
        return this.subfolder;
    }

    public void setSubfolder(String subfolder) {
        this.subfolder = subfolder;
    }

    public static void main(String[] args) throws Exception {
        String assetFile = null;
        String accessKey = null;
        String secretKey = null;
        for (int i = 0; i < args.length; i += 2) {
            if (args[i].startsWith("-a")) {
                accessKey = args[i + 1];
                continue;
            }
            if (args[i].startsWith("-s")) {
                secretKey = args[i + 1];
                continue;
            }
            if (!args[i].startsWith("-f")) continue;
            assetFile = args[i + 1];
        }
        if (accessKey == null || secretKey == null || assetFile == null) {
            System.out.println("Missing arguments - exiting");
            return;
        }
        S3BitStoreService store = new S3BitStoreService();
        BasicAWSCredentials awsCredentials = new BasicAWSCredentials(accessKey, secretKey);
        store.s3Service = new AmazonS3Client((AWSCredentials)awsCredentials);
        Region usEast1 = Region.getRegion((Regions)Regions.US_EAST_1);
        store.s3Service.setRegion(usEast1);
        store.bucketName = "dspace-asset-" + ConfigurationManager.getProperty("dspace.hostname") + ".s3test";
        store.s3Service.createBucket(store.bucketName);
    }
}

