/*
 * Decompiled with CFR 0.152.
 */
package org.dspace.util;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.HttpSolrClient;
import org.apache.solr.client.solrj.request.AbstractUpdateRequest;
import org.apache.solr.client.solrj.request.ContentStreamUpdateRequest;
import org.apache.solr.client.solrj.request.CoreAdminRequest;
import org.apache.solr.client.solrj.request.LukeRequest;
import org.apache.solr.client.solrj.response.CoreAdminResponse;
import org.apache.solr.client.solrj.response.FieldStatsInfo;
import org.apache.solr.client.solrj.response.LukeResponse;
import org.apache.solr.client.solrj.response.RangeFacet;
import org.apache.solr.common.luke.FieldFlag;
import org.apache.solr.common.params.CoreAdminParams;
import org.apache.solr.common.params.SolrParams;
import org.dspace.services.ConfigurationService;
import org.dspace.services.factory.DSpaceServicesFactory;
import org.dspace.util.SolrImportExportException;

public class SolrImportExport {
    private static final ThreadLocal<DateFormat> SOLR_DATE_FORMAT = new ThreadLocal<DateFormat>(){

        @Override
        protected DateFormat initialValue() {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
            simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            return simpleDateFormat;
        }
    };
    private static final ThreadLocal<DateFormat> SOLR_DATE_FORMAT_NO_MS = new ThreadLocal<DateFormat>(){

        @Override
        protected DateFormat initialValue() {
            return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        }
    };
    private static final ThreadLocal<DateFormat> EXPORT_DATE_FORMAT = new ThreadLocal<DateFormat>(){

        @Override
        protected DateFormat initialValue() {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM");
            simpleDateFormat.setTimeZone(TimeZone.getDefault());
            return simpleDateFormat;
        }
    };
    private static final String EXPORT_SEP = "_export_";
    private static final String ACTION_OPTION = "a";
    private static final String CLEAR_OPTION = "c";
    private static final String OVERWRITE_OPTION = "f";
    private static final String DIRECTORY_OPTION = "d";
    private static final String HELP_OPTION = "h";
    private static final String INDEX_NAME_OPTION = "i";
    private static final String KEEP_OPTION = "k";
    private static final String LAST_OPTION = "l";
    public static final int ROWS_PER_FILE = 10000;
    private static final String MULTIPLE_VALUES_SPLITTER = ",";
    private static final Logger log = LogManager.getLogger(SolrImportExport.class);
    private static final ConfigurationService configurationService = DSpaceServicesFactory.getInstance().getConfigurationService();

    private SolrImportExport() {
    }

    public static void main(String[] args) throws org.apache.commons.cli.ParseException {
        DefaultParser parser = new DefaultParser();
        Options options = SolrImportExport.makeOptions();
        try {
            CommandLine line = parser.parse(options, args);
            if (line.hasOption(HELP_OPTION)) {
                SolrImportExport.printHelpAndExit(options, 0);
            }
            String[] indexNames = new String[]{"statistics"};
            if (line.hasOption(INDEX_NAME_OPTION)) {
                indexNames = line.getOptionValues(INDEX_NAME_OPTION);
            } else {
                System.err.println("No index name provided, defaulting to \"statistics\".");
            }
            String directoryName = SolrImportExport.makeDirectoryName(line.getOptionValue(DIRECTORY_OPTION));
            String action = line.getOptionValue(ACTION_OPTION, "export");
            if ("import".equals(action)) {
                for (String indexName : indexNames) {
                    File importDir = new File(directoryName);
                    if (!importDir.exists() || !importDir.canRead()) {
                        System.err.println("Import directory " + directoryName + " doesn't exist or is not readable by the current user. Not importing index " + indexName);
                        continue;
                    }
                    try {
                        String solrUrl = SolrImportExport.makeSolrUrl(indexName);
                        boolean clear = line.hasOption(CLEAR_OPTION);
                        SolrImportExport.importIndex(indexName, importDir, solrUrl, clear);
                    }
                    catch (IOException | SolrServerException | SolrImportExportException e) {
                        System.err.println("Problem encountered while trying to import index " + indexName + ".");
                        e.printStackTrace(System.err);
                    }
                }
            } else if ("export".equals(action)) {
                for (String indexName : indexNames) {
                    String lastValue = line.getOptionValue(LAST_OPTION);
                    File exportDir = new File(directoryName);
                    if (exportDir.exists() && !exportDir.canWrite()) {
                        System.err.println("Export directory " + directoryName + " is not writable by the current user. Not exporting index " + indexName);
                        continue;
                    }
                    if (!exportDir.exists()) {
                        boolean created = exportDir.mkdirs();
                        if (created) continue;
                        System.err.println("Export directory " + directoryName + " could not be created. Not exporting index " + indexName);
                        continue;
                    }
                    try {
                        String solrUrl = SolrImportExport.makeSolrUrl(indexName);
                        String timeField = SolrImportExport.makeTimeField(indexName);
                        SolrImportExport.exportIndex(indexName, exportDir, solrUrl, timeField, lastValue, line.hasOption(OVERWRITE_OPTION));
                    }
                    catch (IOException | SolrServerException | SolrImportExportException e) {
                        System.err.println("Problem encountered while trying to export index " + indexName + ".");
                        e.printStackTrace(System.err);
                    }
                }
            } else if ("reindex".equals(action)) {
                for (String indexName : indexNames) {
                    try {
                        boolean keepExport = line.hasOption(KEEP_OPTION);
                        boolean overwrite = line.hasOption(OVERWRITE_OPTION);
                        SolrImportExport.reindex(indexName, directoryName, keepExport, overwrite);
                    }
                    catch (IOException | SolrServerException | SolrImportExportException e) {
                        e.printStackTrace();
                    }
                }
            } else {
                System.err.println("Unknown action " + action + "; must be import, export or reindex.");
                SolrImportExport.printHelpAndExit(options, 1);
            }
        }
        catch (org.apache.commons.cli.ParseException e) {
            System.err.println("Cannot read command options");
            SolrImportExport.printHelpAndExit(options, 1);
        }
    }

    private static Options makeOptions() {
        Options options = new Options();
        options.addOption(ACTION_OPTION, "action", true, "The action to perform: import, export or reindex. Default: export.");
        options.addOption(CLEAR_OPTION, "clear", false, "When importing, also clear the index first. Ignored when action is export or reindex.");
        options.addOption(OVERWRITE_OPTION, "force-overwrite", false, "When exporting or re-indexing, allow overwrite of existing export files");
        options.addOption(DIRECTORY_OPTION, "directory", true, "The absolute path for the directory to use for import or export. If omitted, [dspace]/solr-export is used.");
        options.addOption(HELP_OPTION, "help", false, "Get help on options for this command.");
        options.addOption(INDEX_NAME_OPTION, "index-name", true, "The names of the indexes to process. At least one is required. Available indexes are: authority, statistics.");
        options.addOption(KEEP_OPTION, "keep", false, "When reindexing, keep the contents of the data export directory. By default, the contents of this directory will be deleted once the reindex has finished. Ignored when action is export or import.");
        options.addOption(LAST_OPTION, "last", true, "When exporting, export records from the last [timeperiod] only. This can be one of: 'd' (beginning of yesterday through to now); 'm' (beginning of the previous month through to end of the previous month); a number, in which case the last [number] of days are exported, through to now (use 0 for today's data). Date calculation is done in UTC. If omitted, all documents are exported.");
        return options;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void reindex(String indexName, String exportDirName, boolean keepExport, boolean overwrite) throws IOException, SolrServerException, SolrImportExportException {
        String tempIndexName = indexName + "-temp";
        String origSolrUrl = SolrImportExport.makeSolrUrl(indexName);
        String baseSolrUrl = StringUtils.substringBeforeLast((String)origSolrUrl, (String)"/");
        String tempSolrUrl = baseSolrUrl + "/" + tempIndexName;
        String instanceIndexName = indexName.startsWith("statistics-") ? "statistics" : indexName;
        String solrInstanceDir = configurationService.getProperty("dspace.dir") + File.separator + "solr" + File.separator + instanceIndexName;
        File solrInstance = new File(solrInstanceDir);
        if (!(solrInstance.exists() && solrInstance.canRead() && solrInstance.isDirectory())) {
            throw new SolrImportExportException("Directory " + solrInstanceDir + "/conf/ doesn't exist or isn't readable. The reindexing process requires the Solr configuration directory for this index to be present on the local machine even if Solr is running on a different host. Not reindexing index " + indexName);
        }
        String timeField = SolrImportExport.makeTimeField(indexName);
        File exportDir = new File(exportDirName);
        boolean createdExportDir = exportDir.mkdirs();
        if (!createdExportDir && !exportDir.exists()) {
            throw new SolrImportExportException("Could not create export directory " + exportDirName);
        }
        if (!exportDir.canWrite()) {
            throw new SolrImportExportException("Can't write to export directory " + exportDirName);
        }
        try {
            HttpSolrClient adminSolr = new HttpSolrClient.Builder(baseSolrUrl).build();
            CoreAdminResponse status = CoreAdminRequest.getStatus((String)indexName, (SolrClient)adminSolr);
            Object coreSizeObj = status.getCoreStatus(indexName).get("sizeInBytes");
            long coreSize = coreSizeObj != null ? Long.valueOf(coreSizeObj.toString()) : -1L;
            long usableExportSpace = exportDir.getUsableSpace();
            if (coreSize >= 0L && usableExportSpace < coreSize) {
                System.err.println("Not enough space in export directory " + exportDirName + "; need at least as much space as the index (" + FileUtils.byteCountToDisplaySize((long)coreSize) + ") but usable space in export directory is only " + FileUtils.byteCountToDisplaySize((long)usableExportSpace) + ". Not continuing with reindex, please use the d option to specify an alternative export directy with sufficient space.");
                return;
            }
            File tempDataDir = new File(configurationService.getProperty("dspace.dir") + File.separator + "temp" + File.separator + "solr-data");
            boolean createdTempDataDir = tempDataDir.mkdirs();
            if (!createdTempDataDir && !tempDataDir.exists()) {
                throw new SolrImportExportException("Could not create temporary data directory " + tempDataDir.getCanonicalPath());
            }
            if (!tempDataDir.canWrite()) {
                throw new SolrImportExportException("Can't write to temporary data directory " + tempDataDir.getCanonicalPath());
            }
            try {
                CoreAdminRequest.Create createRequest = new CoreAdminRequest.Create();
                createRequest.setInstanceDir(solrInstanceDir);
                createRequest.setDataDir(tempDataDir.getCanonicalPath());
                createRequest.setCoreName(tempIndexName);
                ((CoreAdminResponse)createRequest.process((SolrClient)adminSolr)).getStatus();
            }
            catch (SolrServerException e) {
                System.err.println("Caught exception when trying to create temporary core: " + e.getMessage() + "; trying to recover.");
                e.printStackTrace(System.err);
            }
            CoreAdminRequest swapRequest = new CoreAdminRequest();
            swapRequest.setCoreName(indexName);
            swapRequest.setOtherCoreName(tempIndexName);
            swapRequest.setAction(CoreAdminParams.CoreAdminAction.SWAP);
            swapRequest.process((SolrClient)adminSolr);
            try {
                SolrImportExport.exportIndex(indexName, exportDir, tempSolrUrl, timeField, overwrite);
                SolrImportExport.importIndex(indexName, exportDir, tempSolrUrl, true);
            }
            catch (IOException | SolrServerException | SolrImportExportException e) {
                System.err.println("Encountered problem during reindex: " + e.getMessage() + ", will attempt to restore Solr cores");
                e.printStackTrace(System.err);
            }
            HttpSolrClient origSolr = new HttpSolrClient.Builder(origSolrUrl).build();
            origSolr.commit();
            swapRequest = new CoreAdminRequest();
            swapRequest.setCoreName(tempIndexName);
            swapRequest.setOtherCoreName(indexName);
            swapRequest.setAction(CoreAdminParams.CoreAdminAction.SWAP);
            swapRequest.process((SolrClient)adminSolr);
            SolrImportExport.exportIndex(tempIndexName, exportDir, tempSolrUrl, timeField, overwrite);
            SolrImportExport.importIndex(tempIndexName, exportDir, origSolrUrl, false);
            origSolr.commit();
            CoreAdminRequest.unloadCore((String)tempIndexName, (boolean)false, (boolean)false, (SolrClient)adminSolr);
            if (createdTempDataDir && tempDataDir.exists()) {
                FileUtils.deleteDirectory((File)tempDataDir);
            }
        }
        finally {
            if (!keepExport && createdExportDir && exportDir.exists()) {
                FileUtils.deleteDirectory((File)exportDir);
            }
        }
    }

    public static void exportIndex(String indexName, File toDir, String solrUrl, String timeField, boolean overwrite) throws SolrServerException, SolrImportExportException, IOException {
        SolrImportExport.exportIndex(indexName, toDir, solrUrl, timeField, null, overwrite);
    }

    public static void importIndex(final String indexName, File fromDir, String solrUrl, boolean clear) throws IOException, SolrServerException, SolrImportExportException {
        Object[] files;
        if (StringUtils.isBlank((CharSequence)solrUrl)) {
            throw new SolrImportExportException("Could not construct solr URL for index" + indexName + ", aborting export.");
        }
        if (!fromDir.exists() || !fromDir.canRead()) {
            throw new SolrImportExportException("Source directory " + fromDir + " doesn't exist or isn't readable, aborting export of index " + indexName);
        }
        HttpSolrClient solr = new HttpSolrClient.Builder(solrUrl).build();
        List<String> multivaluedFields = SolrImportExport.getMultiValuedFields(solr);
        if (clear) {
            SolrImportExport.clearIndex(solrUrl);
        }
        if ((files = fromDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith(indexName + SolrImportExport.EXPORT_SEP) && name.endsWith(".csv");
            }
        })) == null || files.length == 0) {
            log.warn("No export files found in directory " + fromDir.getCanonicalPath() + " for index " + indexName);
            return;
        }
        Arrays.sort(files);
        for (Object file : files) {
            log.info("Importing file " + ((File)file).getCanonicalPath());
            ContentStreamUpdateRequest contentStreamUpdateRequest = new ContentStreamUpdateRequest("/update/csv");
            contentStreamUpdateRequest.setParam("skip", "_version_");
            for (String mvField : multivaluedFields) {
                contentStreamUpdateRequest.setParam("f." + mvField + ".split", "true");
                contentStreamUpdateRequest.setParam("f." + mvField + ".separator", MULTIPLE_VALUES_SPLITTER);
            }
            contentStreamUpdateRequest.setParam("stream.contentType", "text/csv;charset=utf-8");
            contentStreamUpdateRequest.setAction(AbstractUpdateRequest.ACTION.COMMIT, true, true);
            contentStreamUpdateRequest.addFile((File)file, "text/csv;charset=utf-8");
            solr.request((SolrRequest)contentStreamUpdateRequest);
        }
        solr.commit(true, true);
    }

    private static List<String> getMultiValuedFields(HttpSolrClient solr) {
        ArrayList<String> result = new ArrayList<String>();
        try {
            LukeRequest request = new LukeRequest();
            LukeResponse response = (LukeResponse)request.process((SolrClient)solr);
            Map fields = response.getFieldInfo();
            for (LukeResponse.FieldInfo info : fields.values()) {
                if (!info.getSchema().contains("" + FieldFlag.MULTI_VALUED.getAbbreviation())) continue;
                result.add(info.getName());
            }
        }
        catch (IOException | SolrServerException e) {
            log.fatal("Cannot determine which fields are multi valued: " + e.getMessage(), e);
        }
        return result;
    }

    public static void clearIndex(String solrUrl) throws IOException, SolrServerException {
        HttpSolrClient solr = new HttpSolrClient.Builder(solrUrl).build();
        solr.deleteByQuery("*:*");
        solr.commit();
        solr.optimize();
    }

    public static void exportIndex(String indexName, File toDir, String solrUrl, String timeField, String fromWhen, boolean overwrite) throws SolrServerException, IOException, SolrImportExportException {
        String lastValueFilter;
        log.info(String.format("Export Index [%s] to [%s] using [%s] Time Field[%s] FromWhen[%s]", indexName, toDir, solrUrl, timeField, fromWhen));
        if (StringUtils.isBlank((CharSequence)solrUrl)) {
            throw new SolrImportExportException("Could not construct solr URL for index" + indexName + ", aborting export.");
        }
        if (!toDir.exists() || !toDir.canWrite()) {
            throw new SolrImportExportException("Target directory " + toDir + " doesn't exist or is not writable, aborting export of index " + indexName);
        }
        HttpSolrClient solr = new HttpSolrClient.Builder(solrUrl).build();
        SolrQuery query = new SolrQuery("*:*");
        if (StringUtils.isNotBlank((CharSequence)fromWhen) && StringUtils.isNotBlank((CharSequence)(lastValueFilter = SolrImportExport.makeFilterQuery(timeField, fromWhen)))) {
            query.addFilterQuery(new String[]{lastValueFilter});
        }
        query.setRows(Integer.valueOf(0));
        query.setGetFieldStatistics(timeField);
        Map fieldInfo = solr.query((SolrParams)query).getFieldStatsInfo();
        if (fieldInfo == null || !fieldInfo.containsKey(timeField)) {
            log.warn(String.format("Queried [%s].  No fieldInfo found while exporting index [%s] time field [%s] from [%s]. Export cancelled.", solrUrl, indexName, timeField, fromWhen));
            return;
        }
        FieldStatsInfo timeFieldInfo = (FieldStatsInfo)fieldInfo.get(timeField);
        if (timeFieldInfo == null || timeFieldInfo.getMin() == null) {
            log.warn(String.format("Queried [%s].  No earliest date found while exporting index [%s] time field [%s] from [%s]. Export cancelled.", solrUrl, indexName, timeField, fromWhen));
            return;
        }
        Date earliestTimestamp = (Date)timeFieldInfo.getMin();
        query.setGetFieldStatistics(false);
        query.clearSorts();
        query.setRows(Integer.valueOf(0));
        query.setFacet(true);
        query.add("facet.range", new String[]{timeField});
        query.add("facet.range.start", new String[]{SOLR_DATE_FORMAT.get().format(earliestTimestamp) + "/MONTH"});
        query.add("facet.range.end", new String[]{"NOW/MONTH+1MONTH"});
        query.add("facet.range.gap", new String[]{"+1MONTH"});
        query.setFacetMinCount(1);
        List monthFacets = ((RangeFacet)solr.query((SolrParams)query).getFacetRanges().get(0)).getCounts();
        for (RangeFacet.Count monthFacet : monthFacets) {
            Date monthStartDate;
            String monthStart = monthFacet.getValue();
            try {
                monthStartDate = SOLR_DATE_FORMAT_NO_MS.get().parse(monthStart);
            }
            catch (ParseException e) {
                throw new SolrImportExportException("Could not read start of month batch as date: " + monthStart, e);
            }
            int docsThisMonth = monthFacet.getCount();
            SolrQuery monthQuery = new SolrQuery("*:*");
            monthQuery.setRows(Integer.valueOf(10000));
            monthQuery.set("wt", new String[]{"csv"});
            monthQuery.set("fl", new String[]{"*"});
            monthQuery.setParam("csv.mv.separator", new String[]{MULTIPLE_VALUES_SPLITTER});
            monthQuery.addFilterQuery(new String[]{timeField + ":[" + monthStart + " TO " + monthStart + "+1MONTH]"});
            for (int i = 0; i < docsThisMonth; i += 10000) {
                String message;
                monthQuery.setStart(Integer.valueOf(i));
                URL url = new URL(solrUrl + "/select?" + monthQuery.toString());
                File file = new File(toDir.getCanonicalPath(), SolrImportExport.makeExportFilename(indexName, monthStartDate, docsThisMonth, i));
                if (!file.createNewFile() && !overwrite) {
                    if (file.exists()) {
                        message = String.format("Solr export file [%s] already exists.  Export failed for Index [%s] Month [%s] Batch [%d] Num Docs [%d]", file.getCanonicalPath(), indexName, monthStart, i, docsThisMonth);
                        throw new SolrImportExportException(message);
                    }
                    message = String.format("Cannot create solr export file [%s].  Export failed for Index [%s] Month [%s] Batch [%d] Num Docs [%d]", file.getCanonicalPath(), indexName, monthStart, i, docsThisMonth);
                    throw new SolrImportExportException(message);
                }
                FileUtils.copyURLToFile((URL)url, (File)file);
                message = String.format("Solr export to file [%s] complete.  Export for Index [%s] Month [%s] Batch [%d] Num Docs [%d]", file.getCanonicalPath(), indexName, monthStart, i, docsThisMonth);
                log.info(message);
            }
        }
    }

    private static String makeFilterQuery(String timeField, String lastValue) {
        if ("m".equals(lastValue)) {
            return timeField + ":[NOW/MONTH-1MONTH TO NOW/MONTH]";
        }
        int days = DIRECTORY_OPTION.equals(lastValue) ? 1 : Integer.valueOf(lastValue);
        return timeField + ":[NOW/DAY-" + days + "DAYS TO " + SOLR_DATE_FORMAT.get().format(new Date()) + "]";
    }

    private static String makeDirectoryName(String directoryValue) {
        if (StringUtils.isNotBlank((CharSequence)directoryValue)) {
            return directoryValue;
        }
        return configurationService.getProperty("dspace.dir") + File.separator + "solr-export" + File.separator;
    }

    private static String makeExportFilename(String indexName, Date exportStart, long totalRecords, int index) {
        String exportFileNumber = "";
        if (totalRecords > 10000L) {
            exportFileNumber = StringUtils.leftPad((String)("" + index / 10000), (int)((int)Math.ceil(Math.log10(totalRecords / 10000L))), (String)"0");
        }
        return indexName + EXPORT_SEP + EXPORT_DATE_FORMAT.get().format(exportStart) + (String)(StringUtils.isNotBlank((CharSequence)exportFileNumber) ? "_" + exportFileNumber : "") + ".csv";
    }

    private static String makeSolrUrl(String indexName) {
        if (indexName.startsWith("statistics")) {
            return configurationService.getProperty("solr-statistics.server") + indexName.replaceFirst("statistics", "");
        }
        if ("authority".equals(indexName)) {
            return configurationService.getProperty("solr.authority.server");
        }
        return "http://localhost:8080/solr/" + indexName;
    }

    private static String makeTimeField(String indexName) {
        if (indexName.startsWith("statistics")) {
            return "time";
        }
        if ("authority".equals(indexName)) {
            return "last_modified_date";
        }
        return null;
    }

    private static void printHelpAndExit(Options options, int exitCode) {
        HelpFormatter myhelp = new HelpFormatter();
        myhelp.printHelp(SolrImportExport.class.getSimpleName() + "\n", options);
        System.out.println("\n\nCommand Defaults");
        System.out.println("\tsolr-export-statistics  [-a export]  [-i statistics]");
        System.out.println("\tsolr-import-statistics  [-a import]  [-i statistics]");
        System.out.println("\tsolr-reindex-statistics [-a reindex] [-i statistics]");
        System.exit(exitCode);
    }
}

