/**
 * The contents of this file are subject to the license and copyright
 * detailed in the LICENSE and NOTICE files at the root of the source
 * tree and available online at
 *
 * http://www.dspace.org/license/
 */
package org.dspace.eperson.dao.impl;

import java.sql.SQLException;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import jakarta.persistence.Query;
import jakarta.persistence.criteria.CriteriaBuilder;
import jakarta.persistence.criteria.CriteriaQuery;
import jakarta.persistence.criteria.Predicate;
import jakarta.persistence.criteria.Root;
import org.apache.commons.lang3.tuple.Pair;
import org.dspace.core.AbstractHibernateDAO;
import org.dspace.core.Context;
import org.dspace.eperson.Group;
import org.dspace.eperson.Group2GroupCache;
import org.dspace.eperson.Group2GroupCache_;
import org.dspace.eperson.dao.Group2GroupCacheDAO;

/**
 * Hibernate implementation of the Database Access Object interface class for the Group2GroupCache object.
 * This class is responsible for all database calls for the Group2GroupCache object and is autowired by Spring.
 * This class should never be accessed directly.
 *
 * @author kevinvandevelde at atmire.com
 */
public class Group2GroupCacheDAOImpl extends AbstractHibernateDAO<Group2GroupCache> implements Group2GroupCacheDAO {
    protected Group2GroupCacheDAOImpl() {
        super();
    }

    @Override
    public Set<Pair<UUID, UUID>> getCache(Context context) throws SQLException {
        Query query = createQuery(
            context,
            "SELECT new org.apache.commons.lang3.tuple.ImmutablePair(g.parent.id, g.child.id) FROM Group2GroupCache g"
        );
        List<Pair<UUID, UUID>> results = query.getResultList();
        return new HashSet<Pair<UUID, UUID>>(results);
    }

    @Override
    public List<Group2GroupCache> findByParent(Context context, Group group) throws SQLException {
        CriteriaBuilder criteriaBuilder = getCriteriaBuilder(context);
        CriteriaQuery criteriaQuery = getCriteriaQuery(criteriaBuilder, Group2GroupCache.class);
        Root<Group2GroupCache> group2GroupCacheRoot = criteriaQuery.from(Group2GroupCache.class);
        criteriaQuery.select(group2GroupCacheRoot);
        criteriaQuery.where(criteriaBuilder.equal(group2GroupCacheRoot.get(Group2GroupCache_.parent), group));
        return list(context, criteriaQuery, true, Group2GroupCache.class, -1, -1);
    }

    @Override
    public List<Group2GroupCache> findByChildren(Context context, Iterable<Group> groups) throws SQLException {
        CriteriaBuilder criteriaBuilder = getCriteriaBuilder(context);
        CriteriaQuery criteriaQuery = getCriteriaQuery(criteriaBuilder, Group2GroupCache.class);
        Root<Group2GroupCache> group2GroupCacheRoot = criteriaQuery.from(Group2GroupCache.class);
        List<Predicate> eqPredicates = new LinkedList<>();
        for (Group group : groups) {
            eqPredicates.add(criteriaBuilder.equal(group2GroupCacheRoot.get(Group2GroupCache_.child), group));
        }
        Predicate orPredicate = criteriaBuilder.or(eqPredicates.toArray(new Predicate[] {}));
        criteriaQuery.select(group2GroupCacheRoot);
        criteriaQuery.where(orPredicate);
        return list(context, criteriaQuery, true, Group2GroupCache.class, -1, -1);
    }

    @Override
    public Group2GroupCache findByParentAndChild(Context context, Group parent, Group child) throws SQLException {
        Query query = createQuery(context,
                                  "FROM Group2GroupCache g WHERE g.parent = :parentGroup AND g.child = :childGroup");

        query.setParameter("parentGroup", parent);
        query.setParameter("childGroup", child);
        query.setHint("org.hibernate.cacheable", Boolean.TRUE);

        return singleResult(query);
    }

    @Override
    public Group2GroupCache find(Context context, Group parent, Group child) throws SQLException {
        CriteriaBuilder criteriaBuilder = getCriteriaBuilder(context);
        CriteriaQuery criteriaQuery = getCriteriaQuery(criteriaBuilder, Group2GroupCache.class);
        Root<Group2GroupCache> group2GroupCacheRoot = criteriaQuery.from(Group2GroupCache.class);
        criteriaQuery.select(group2GroupCacheRoot);
        criteriaQuery.where(
            criteriaBuilder.and(criteriaBuilder.equal(group2GroupCacheRoot.get(Group2GroupCache_.parent), parent),
                                criteriaBuilder.equal(group2GroupCacheRoot.get(Group2GroupCache_.child), child)
            )
        );
        return uniqueResult(context, criteriaQuery, true, Group2GroupCache.class);
    }

    @Override
    public void deleteAll(Context context) throws SQLException {
        createQuery(context, "delete from Group2GroupCache").executeUpdate();
    }

    @Override
    public void deleteFromCache(Context context, UUID parent, UUID child) throws SQLException {
        Query query = getHibernateSession(context).createNativeQuery(
            "delete from group2groupcache g WHERE g.parent_id = :parent AND g.child_id = :child"
        );
        query.setParameter("parent", parent);
        query.setParameter("child", child);
        query.executeUpdate();
    }

    @Override
    public void addToCache(Context context, UUID parent, UUID child) throws SQLException {
        Query query = getHibernateSession(context).createNativeQuery(
            "insert into group2groupcache (parent_id, child_id) VALUES (:parent, :child)"
        );
        query.setParameter("parent", parent);
        query.setParameter("child", child);
        query.executeUpdate();
    }
}
