/**
 * The contents of this file are subject to the license and copyright
 * detailed in the LICENSE and NOTICE files at the root of the source
 * tree and available online at
 *
 * http://www.dspace.org/license/
 */
package org.dspace.app.rest.authorization;

import org.dspace.app.rest.model.BaseObjectRest;
import org.dspace.app.rest.model.RestAddressableModel;
import org.dspace.content.Site;
import org.dspace.eperson.EPerson;

/**
 * An authorization is the right for a specific {@link EPerson}, eventually null to indicate unauthenticated users, to
 * use a specific {@link AuthorizationFeature} on a defined object. The target object must implement the
 * {@link RestAddressableModel} interface so to be directly addressable
 *
 * @author Andrea Bollini (andrea.bollini at 4science.it)
 */
public class Authorization {
    /**
     * the user authorized to use the feature. <code>null</code> mean unauthenticated user
     */
    private EPerson eperson;

    /**
     * the feature that is authorized to be used
     */
    private AuthorizationFeature feature;

    /**
     * the object where the feature can be used. Not null, for repository wide feature use the {@link Site} object
     */
    private BaseObjectRest object;

    public Authorization() {
    }

    public Authorization(EPerson eperson, AuthorizationFeature feature, BaseObjectRest object) {
        super();
        this.eperson = eperson;
        this.feature = feature;
        this.object = object;
    }


    /**
     * 
     * @return the user authorized to use the feature. <code>null</code> mean unauthenticated user
     */
    public EPerson getEperson() {
        return eperson;
    }

    /**
     * 
     * @param eperson
     *            the user authorized to use the feature. <code>null</code> mean unauthenticated user
     */
    public void setEperson(EPerson eperson) {
        this.eperson = eperson;
    }

    /**
     * 
     * @return the feature that is authorized to be used
     */
    public AuthorizationFeature getFeature() {
        return feature;
    }

    /**
     * 
     * @param feature
     *            the feature that is authorized to be used
     */
    public void setFeature(AuthorizationFeature feature) {
        this.feature = feature;
    }

    /**
     * 
     * @return the object where the feature can be used. Not null, for repository wide feature use the {@link Site}
     *         object
     */
    public BaseObjectRest getObject() {
        return object;
    }

    /**
     * 
     * @param object
     *            the object where the feature can be used. Not null, for repository wide feature use the {@link Site}
     *            object
     */
    public void setObject(BaseObjectRest object) {
        this.object = object;
    }


    /**
     * 
     * @return an unique business identifier generated by concatenation of eperson uuid (if any), feature name and
     *         object unique identifier.
     *         Some examples
     *         alwaystrue_core.site_94274020-e617-43b8-90e6-277d04f6be13
     *         8c7b9132-eadc-4199-af6d-3260a678e96f_alwaystrueadmins_core.site_94274020-e617-43b8-90e6-277d04f6be13
     *         8c7b9132-eadc-4199-af6d-3260a678e96f_withdrawItem_core.item_c8924526-67ef-479a-8e37-dd8d19e625e9
     *         8c7b9132-eadc-4199-af6d-3260a678e96f_alwaystrue_submission.workspaceitem_1
     */
    public String getID() {
        StringBuilder sb = new StringBuilder();
        if (eperson != null) {
            sb.append(eperson.getID().toString()).append("_");
        }
        sb.append(feature.getName()).append("_").append(object.getUniqueType()).append("_").append(object.getId());
        return sb.toString();
    }

}