/*
 * Copyright (C) 2018 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.common.algorithms;

import dulab.adap.datamodel.BetterPeak;
import dulab.adap.datamodel.Chromatogram;

public class PeakUtils {

    public static double similarity(BetterPeak peak1, BetterPeak peak2, double retTimeTolerance) {

        if (java.lang.Math.abs(peak1.getRetTime() - peak2.getRetTime()) > retTimeTolerance)
            return 0.0;

        Chromatogram chromatogram1 = peak1.chromatogram;
        Chromatogram chromatogram2 = peak2.chromatogram;

        double prevRetTime = Double.NaN;
        double prevProduct = Double.NaN;
        int index1 = 0;
        int index2 = 0;
        double dotProduct = 0.0;

        while (index1 < chromatogram1.length || index2 < chromatogram2.length) {

            double t1 = index1 < chromatogram1.length ? chromatogram1.xs[index1] : Double.MAX_VALUE;
            double t2 = index2 < chromatogram2.length ? chromatogram2.xs[index2] : Double.MAX_VALUE;

            double retTime;
            double product;

            if (t1 == t2) {
                retTime = t1;
                product = chromatogram1.ys[index1] * chromatogram2.ys[index2];
                ++index1;
                ++index2;
            } else if (t1 < t2) {
                retTime = t1;
                product = chromatogram1.ys[index1] * chromatogram2.interpolateIntensity(t1);
                ++index1;
            } else {  // t2 < t1
                retTime = t2;
                product = chromatogram1.interpolateIntensity(t2) * chromatogram2.ys[index2];
                ++index2;
            }

            if (!Double.isNaN(prevRetTime) && !Double.isNaN(prevProduct))
                dotProduct += 0.5 * (prevProduct + product) * (retTime - prevRetTime);

            prevRetTime = retTime;
            prevProduct = product;
        }

        return dotProduct / peak1.chromatogram.norm / peak2.chromatogram.norm;
    }
}
