/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.common.algorithms;

import com.google.common.base.CharMatcher;
import com.google.common.collect.Range;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */


public class String 
{
    /**
     * Parse comma-separated and dash-separated text to a list of ranges
     * @param text
     * @return list of ranges
     */
    
    public static List <Range <Double>> toRanges(final java.lang.String text)
    {
        final java.lang.String ERROR_MESSAGE = "Cannot parse the string";
        
        final java.lang.String ALLOWED_CHARACTERS = "1234567890.,- ";
        final java.lang.String WHITESPACE = " ";
        final java.lang.String COMMA = ",";
        final java.lang.String DASH = "-";
        
        java.lang.String filteredText = CharMatcher.anyOf(ALLOWED_CHARACTERS).retainFrom(text);
        
        if (!filteredText.equals(text))
            throw new IllegalArgumentException(ERROR_MESSAGE + ": the string contains unknown characters");
        
        // -------------------------------------------------------------
        // Remove whitespace and parse comma-separated parts of the text
        // -------------------------------------------------------------
        
        List <Range <Double>> result = new ArrayList <> ();
        
        if (filteredText.isEmpty()) return result;
        
        filteredText = CharMatcher.anyOf(WHITESPACE).removeFrom(filteredText);
        
        try 
        {
            for (java.lang.String strRange : filteredText.split(COMMA))
            {
                java.lang.String[] strValues = strRange.split(DASH);
                
                switch (strValues.length) 
                {
                    case 1:
                        double value = Double.parseDouble(strValues[0]);
                        result.add(Range.singleton(value));
                        break;
                    
                    case 2:
                        double start = Double.parseDouble(strValues[0]);
                        double end = Double.parseDouble(strValues[1]);
                        result.add(Range.closed(start, end));
                        break;
                }
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException(ERROR_MESSAGE + ": " + e.getMessage());
        }
        
        return result;
    }
    
    /**
     * Parse a list of ranges to a string
     * @param ranges
     * @return a string
     */
    
    public static java.lang.String fromRanges(
            final List <Range <Double>> ranges)
    {
        java.lang.String result = "";
        
        for (Range <Double> range : ranges) 
        {
            double start = range.lowerEndpoint();
            double end = range.upperEndpoint();
            
            if (start == end) 
                result += Double.toString(start) + ",";
            else
                result += Double.toString(start) + "-" + Double.toString(end) + ",";
        }
        
        // Remove the last comma (if any)
        
        if (result.isEmpty()) return result;
        
        int endPosition = result.length() - 1;
        
        if (result.substring(endPosition).equals(","))
            result = result.substring(0, endPosition);
        
        return result;
    }
}
