/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.algorithms.machineleanring;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.math3.ml.clustering.Cluster;
import org.apache.commons.math3.ml.clustering.Clusterable;
import org.apache.commons.math3.ml.clustering.DBSCANClusterer;
import org.apache.commons.math3.ml.clustering.DoublePoint;

/**
 *
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */


public class ApacheDBSCANClustering 
{
    
    private static class ObjectWrapper implements Clusterable
    {
        private final double[] position;
        private final Object object;
        
        public ObjectWrapper(double position, Object object) {
            this.position = new double[] {position};
            this.object = object;
        }
        
        @Override
        public double[] getPoint() {
            return position;
        }
        
        public Object getObject() {
            return object;
        }
    }
    
    /**
     * Performs clustering using DBSCAN algorithm
     * 
     * @param values position of an object. Objects are clustered based on the
     *               distances between corresponding values
     * 
     * @param eps distance that defines the neighborhood of an object
     * 
     * @param minObjects minimum number of density-connected objects required to
     *                   form a cluster
     * 
     * @param withNoise if true, the outlier (noise) objects will be added as
     *                  separate clusters. Otherwise, they are skipped
     * 
     * @return list of clusters
     */
    
    public static List <List <Double>> call(final List <Double> values,
            final List <Object> objects,
            double eps, int minObjects, boolean withNoise)
    {   
        List <DoublePoint> clusterInput = new ArrayList <> (values.size());
        for (double v : values)
            clusterInput.add(new DoublePoint(new double[] {v}));
        
        DBSCANClusterer <DoublePoint> clusterer = 
                new DBSCANClusterer <> (eps, minObjects);
        
        List <Cluster <DoublePoint>> clusterOutput =
                clusterer.cluster(clusterInput);
        
        List <List <Double>> result = new ArrayList <> (clusterOutput.size());
        
        for (Cluster <DoublePoint> c : clusterOutput) 
        {
            List <DoublePoint> points = c.getPoints();
            
            List <Double> cluster = new ArrayList <> (points.size());
            for (DoublePoint pt : points)
                cluster.add(pt.getPoint()[0]);
            
            result.add(cluster);
        }
        
        if (withNoise) addOutlierObjects(values, result);
        
        return result;
    }
    
    /**
     * Find the values that do not belong to any of the clusters and add them as
     * separate clusters
     * 
     * @param values original values to cluster
     * 
     * @param clusters clusters returned from DBSCANClusterer
     */
    
    private static void addOutlierObjects(
            List <Double> values, List <List <Double>> clusters)
    {
        Set <Double> clusteredObjects = new HashSet <> ();
        for (List <Double> c : clusters)
            clusteredObjects.addAll(c);
        
        for (Double v : values)
            if (!clusteredObjects.contains(v)) {
                List <Double> cluster = new ArrayList <> (1);
                cluster.add(v);
                clusters.add(cluster);
            }
    }
    
}
