/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.algorithms.machineleanring;

import org.apache.commons.math3.ml.clustering.Cluster;
import org.apache.commons.math3.ml.clustering.Clusterable;
import org.apache.commons.math3.ml.clustering.DBSCANClusterer;
import org.apache.commons.math3.ml.clustering.DoublePoint;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 *
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */


public class ApacheDBSCANClusteringV2 <ObjectType>
{
    
    private class ObjectWrapper implements Clusterable
    {
        private final double[] position;
        private final ObjectType object;
        
        public ObjectWrapper(double position, ObjectType object) {
            this.position = new double[] {position};
            this.object = object;
        }
        
        @Override
        public double[] getPoint() {
            return position;
        }
        
        public ObjectType getObject() {
            return object;
        }
    }
    
    /**
     * Performs clustering using DBSCAN algorithm
     * 
     * @param values position of an object. Objects are clustered based on the
     *               distances between corresponding values
     * 
     * @param eps distance that defines the neighborhood of an object
     * 
     * @param minObjects minimum number of density-connected objects required to
     *                   form a cluster
     * 
     * @param withNoise if true, the outlier (noise) objects will be added as
     *                  separate clusters. Otherwise, they are skipped
     * 
     * @return list of clusters
     */
    
    public List <List <ObjectType>> call(final List <Double> values,
            final List <ObjectType> objects,
            double eps, int minObjects, boolean withNoise)
    {
        final int size = values.size();
        List<ObjectWrapper> clusterInput = new ArrayList <> (values.size());
        for (int i = 0; i < size; ++i)
            clusterInput.add(new ObjectWrapper(values.get(i), objects.get(i)));
        
        DBSCANClusterer<ObjectWrapper> clusterer =
                new DBSCANClusterer <> (eps, minObjects);
        
        List <Cluster <ObjectWrapper>> clusterOutput =
                clusterer.cluster(clusterInput);
        
        List <List <ObjectType>> result = new ArrayList <> (clusterOutput.size());
        
        for (Cluster<ObjectWrapper> c : clusterOutput)
        {
            List<ObjectWrapper> points = c.getPoints();
            
            List<ObjectType> cluster = new ArrayList <> (points.size());
            for (ObjectWrapper pt : points) cluster.add(pt.getObject());
            
            result.add(cluster);
        }
        
        if (withNoise) addOutlierObjects(objects, result);
        
        return result;
    }
    
    /**
     * Find the values that do not belong to any of the clusters and add them as
     * separate clusters
     * 
     * @param objects original objects to cluster
     * 
     * @param clusters clusters returned from DBSCANClusterer
     */
    
    private void addOutlierObjects(
            List <ObjectType> objects, List <List <ObjectType>> clusters)
    {
        Set <ObjectType> clusteredObjects = new HashSet <> ();
        for (List <ObjectType> c : clusters) clusteredObjects.addAll(c);
        
        for (ObjectType o : objects)
            if (!clusteredObjects.contains(o)) {
                List <ObjectType> cluster = new ArrayList <> (1);
                cluster.add(o);
                clusters.add(cluster);
            }
    }
    
}
