/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.algorithms.machineleanring;

import dulab.adap.common.algorithms.ArgMin;
import dulab.adap.common.algorithms.SymmetricMatrix;
import java.util.ArrayList;
import java.util.List;

/**
 * This class contains optimization and Clustering algorithms
 * 
 * @author aleksandrsmirnov
 */
public class Clustering <T> {
    
    /**
     * Cluster objects using complete-linkage hierarchical Clustering algorithm
     * 
     * @param values list of numbers corresponding to the objects. 
     *               The algorithms uses euclidean distances between these 
     *               values
     * 
     * @param objects objects to cluster
     * 
     * @param maxDistance maximum size of one cluster
     * 
     * @return list of clusters
     */
    
    public List <List <T>> hcluster (
            final List <Double> values, 
            final List <T> objects,
            double maxDistance)
    {
        final double infinity = Double.MAX_VALUE;
        int size = values.size();
        
        SymmetricMatrix distances = new SymmetricMatrix(size);
        for (int i = 0; i < size; ++i) {
            distances.set(i, i, infinity);
            for (int j = 0; j < i; ++j)
                distances.set(i, j,
                        java.lang.Math.abs(values.get(i) - values.get(j)));
        }
        
        return hcluster(distances, objects, maxDistance);
    }
    
    /**
     * Cluster objects using complete-linkage hierarchical Clustering algorithm
     * 
     * @param matrix Symmetric matrix of distances between objects
     * 
     * @param objects objects to cluster
     * 
     * @param maxDistance maximum size of one cluster
     * 
     * @return list of clusters
     */
    
    public List <List<T>> hcluster (
            final SymmetricMatrix matrix, 
            final List <T> objects,
            double maxDistance)
    {
        int size = matrix.size();
        if (size != objects.size())
            throw new IllegalArgumentException(
                    "1st and 2nd arguments should have the same size");
        
        SymmetricMatrix distances = new SymmetricMatrix(matrix);
        
        // Create a collection of single-element clusters
        List <List <T>> clusters = new ArrayList <> ();
        for (int i = 0; i < size; ++i) {
            clusters.add(new ArrayList <T> ());
            clusters.get(i).add(objects.get(i));
        }
        
        
        while (distances.size() > 1) {
            size = distances.size();
            
            ArgMin min = distances.argmin();
            
            if (min.value > maxDistance) break;
            
            // Calculate new distances
            for (int i = 0; i < size; ++i)
                if (i != min.index1 && i != min.index2) {
                    double v = java.lang.Math.max(
                            distances.get(min.index1, i),
                            distances.get(min.index2, i));
                    
                    distances.set(min.index1, i, v);
                    distances.set(min.index2, i, v);
                }
            
            SymmetricMatrix new_distances = new SymmetricMatrix(size - 1);
            List <List <T>> new_clusters = new ArrayList <> ();
            for (int i = 0; i < size - 1; ++i) 
                new_clusters.add(new ArrayList <T> ());
            
            for (int i = 0; i < size; ++i) {
                int inew = i;
                if (i > min.index2) --inew;
                
                // Update distance matrix
                for (int j = i; j < size; ++j) {
                    int jnew = j;
                    if (j > min.index2) --jnew;
                    if (jnew < size - 1)
                        new_distances.set(inew, jnew, distances.get(i, j));
                }
                
                // Update clusters
                if (i == min.index2)
                    new_clusters.get(min.index1).addAll(clusters.get(i));
                else
                    new_clusters.get(inew).addAll(clusters.get(i));
            }
            
            distances = new SymmetricMatrix(new_distances);
            
            clusters = new ArrayList <> ();
            for (int i = 0; i < size - 1; ++i) {
                clusters.add(new ArrayList <T> ());
                clusters.get(i).addAll(new_clusters.get(i));
            }
        }
        
        return clusters;
    }
    
    
}