/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.common.algorithms.machineleanring;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import org.jgrapht.Graph;
import org.jgrapht.UndirectedGraph;
import org.jgrapht.alg.StoerWagnerMinimumCut;
import org.jgrapht.graph.DefaultEdge;
import org.jgrapht.graph.SimpleGraph;
import org.jgrapht.graph.Subgraph;
import org.jgrapht.graph.UndirectedSubgraph;

/**
 *
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */


public class JGraphTClustering 
{
    /**
     * Cluster the list of objects as follows:
     *     1. Construct a graph of the objects so that the similarity of
     *        connected objects exceeds the threshold
     *     2. Find highly-connected subgraphs using HCS-algorithm
     *     3. Form clusters of the highly connected subgraphs
     * 
     * @param objects list of objects
     * @param similarity similarity-matrix
     * @param threshold value
     * @return list of clusters
     */
    
    public static Set <Set <Object>> call(final List <Object> objects,
            final double[][] similarity,
            final double threshold) 
    {
        dimensionCheck(objects, similarity);
        
        UndirectedGraph <Object, DefaultEdge> graph =
                buildGraph(objects, similarity, threshold);
        
        List <UndirectedGraph <Object, DefaultEdge>> clusters = clusterGraph(graph);
        
        Set <Set <Object>> result = new HashSet <> (clusters.size());
        for (Graph <Object, DefaultEdge> cluster : clusters)
            result.add(new HashSet <> (cluster.vertexSet()));
        
        return result;
    }
    
    /**
     * Cluster the list of objects as in Call and finds the most connected node 
     * in each cluster
     * 
     * @param objects list of objects
     * @param similarity similarity-matrix
     * @param threshold value
     * @return list of the most connected objects
     */
    
    public static Set <Set <Object>> highestDegreeNodes(final List <Object> objects,
            final double[][] similarity,
            final double threshold) 
    {
        dimensionCheck(objects, similarity);
        
        Set <Set <Object>> result = new HashSet <> ();
        
        UndirectedGraph <Object, DefaultEdge> graph =
                buildGraph(objects, similarity, threshold);
        
        List <UndirectedGraph <Object, DefaultEdge>> clusters = 
                clusterGraph(graph);
        
        // ---------------------------------------------------------------
        // For each cluster, create a sorted map of (degree, nodes)-pairs.
        // Then, save the nodes of the highest degree.
        // ---------------------------------------------------------------
        
        NavigableMap <Integer, Set> sortedNodes = new TreeMap <> ();
        for (UndirectedGraph <Object, DefaultEdge> cluster : clusters)
        {
            sortedNodes.clear();
            
            for (Object node : cluster.vertexSet()) 
            {
                int degree = cluster.degreeOf(node);
                Set <Object> nodes = sortedNodes.get(degree);
                if (nodes != null)
                    nodes.add(node);
                else {
                    nodes = new HashSet <> ();
                    nodes.add(node);
                    sortedNodes.put(degree, nodes);
                }
            }
            
            result.add(sortedNodes.lastEntry().getValue());
        }
        
        return result;
    }
    
    /**
     * Check if the size of the objects and the distance matrix do not coincide
     * @param objects list of objects
     * @param distances similarity matrix
     * @throws IllegalArgumentException 
     */
    
    private static void dimensionCheck(final List <Object> objects,
            final double[][] similarity)
            throws IllegalArgumentException
    {
        final String message = "Number of objects does not match the size of the distance matrix";
        
        int size = objects.size();
        
        if (size != similarity.length)
            throw new IllegalArgumentException(message);
        
        for (double[] row : similarity)
            if (size != row.length)
                throw new IllegalArgumentException(message);
    }
    
    /**
     * Create a undirected graph of objects so that the similarity of
     *     connected objects exceeds the threshold
     * @param objects list of objects
     * @param similarityMatrix similarity-matrix
     * @param threshold value
     * @return a undirected graph
     */
    
    private static UndirectedGraph <Object, DefaultEdge> buildGraph(
            final List <Object> objects,
            final double[][] similarityMatrix,
            final double threshold)
    {
        UndirectedGraph <Object, DefaultEdge> graph = 
                new SimpleGraph <> (DefaultEdge.class);
        
        for (Object object : objects) graph.addVertex(object);
        
        int size = objects.size();
        
        for (int i = 0; i < size; ++i)
            for (int j = i + 1; j < size; ++j) 
            {   
                double similarity = similarityMatrix[i][j];
             
                // Symmetry check
                if (similarity != similarityMatrix[j][i])
                    throw new IllegalArgumentException("Similarity-matrix is not symmetric");
                
                if (similarity > threshold)
                    graph.addEdge(objects.get(i), objects.get(j));
            }
        
        return graph;
    }
    
//    private static List <UndirectedGraph <Object, DefaultEdge>> clusterGraph(
//            final UndirectedGraph <Object, DefaultEdge> graph)
//    {
//        List <UndirectedGraph <Object, DefaultEdge>> result = new ArrayList <> ();
//        
//        HCS(graph, result);
//        
//        return result;
//    }
    
    /**
     * Find densely connected subgraphs
     * 
     * @param graph
     * @return list of subgraphs
     */
    
    private static List <UndirectedGraph <Object, DefaultEdge>> clusterGraph(
            final UndirectedGraph <Object, DefaultEdge> graph)
    {
        List <UndirectedGraph <Object, DefaultEdge>> result = new ArrayList <> ();
        
        List <UndirectedGraph <Object, DefaultEdge>> layer = new ArrayList <> ();
        layer.add(graph);
        
        List <UndirectedGraph <Object, DefaultEdge>> newLayer = new ArrayList <> ();
        
        Set <Object> vertices1;
        Set <Object> vertices2 = new HashSet <> ();
        Set <DefaultEdge> cut = new HashSet <> ();
        
        while (!layer.isEmpty())
        {
            for (UndirectedGraph <Object, DefaultEdge> g : layer)
            {
                if (g.vertexSet().size() <= 1) continue;
                
                vertices2.clear();
                cut.clear();
                
                StoerWagnerMinimumCut <Object, DefaultEdge> minCut = 
                        new StoerWagnerMinimumCut <> (g);
                
                vertices1 = minCut.minCut();
                vertices2.addAll(g.vertexSet());
                vertices2.removeAll(vertices1);
                
                Subgraph <Object, DefaultEdge, UndirectedGraph <Object, DefaultEdge>> 
                        subgraph1 = new Subgraph <> (graph, vertices1);
        
                Subgraph <Object, DefaultEdge, UndirectedGraph <Object, DefaultEdge>> 
                        subgraph2 = new Subgraph <> (graph, vertices2);
                
                cut.addAll(g.edgeSet());
                cut.removeAll(subgraph1.edgeSet());
                cut.removeAll(subgraph2.edgeSet());
                
                if (cut.size() >= g.vertexSet().size() / 2) // Highly Connected Graph
                    result.add(g);
                else 
                {
                    newLayer.add(new UndirectedSubgraph <> (g, subgraph1.vertexSet(), 
                            subgraph1.edgeSet()));
                    
                    newLayer.add(new UndirectedSubgraph <> (g, subgraph2.vertexSet(), 
                            subgraph2.edgeSet()));
                }
            }
            
            layer.clear();
            layer.addAll(newLayer);
            newLayer.clear();
        }
        
        return result;
    }
    
//    private static void HCS(UndirectedGraph <Object, DefaultEdge> graph,
//            List <UndirectedGraph <Object, DefaultEdge>> clusters)
//    {
//        if (graph.vertexSet().size() <= 1) return;
//        
//        StoerWagnerMinimumCut <Object, DefaultEdge> minCut = 
//                new StoerWagnerMinimumCut <> (graph);
//        
//        Set <Object> vertices1 = minCut.minCut();
//        Set <Object> vertices2 = new HashSet <> (graph.vertexSet());
//        vertices2.removeAll(vertices1);
//        
//        Subgraph <Object, DefaultEdge, UndirectedGraph <Object, DefaultEdge>> 
//                subgraph1 = new Subgraph <> (graph, vertices1);
//        
//        Subgraph <Object, DefaultEdge, UndirectedGraph <Object, DefaultEdge>> 
//                subgraph2 = new Subgraph <> (graph, vertices2);
//        
//        Set <DefaultEdge> cut = new HashSet <> (graph.edgeSet());
//        cut.removeAll(subgraph1.edgeSet());
//        cut.removeAll(subgraph2.edgeSet());
//        
//        if (cut.size() > graph.vertexSet().size() * 0.4) // Highly Connected Graph
//            clusters.add(graph);
//        else 
//        {
//            // -----------------
//            // Create Subgraph 1
//            // -----------------
//            
//            HCS(new UndirectedSubgraph <> (graph, subgraph1.vertexSet(), 
//                    subgraph1.edgeSet()), clusters);
//            
//            // -----------------
//            // Create Subgraph 2
//            // -----------------
//            
//            HCS(new UndirectedSubgraph <> (graph, subgraph2.vertexSet(), 
//                    subgraph2.edgeSet()), clusters);
//            
////            UndirectedGraph <Object, DefaultEdge> graph2 = 
////                    new SimpleGraph <> (subgraph2.getEdgeFactory());
////            
////            HCS(graph2, clusters);
//        }
//    }
}
