package dulab.adap.common.algorithms.optimization;

import com.google.common.collect.Range;
import com.joptimizer.functions.ConvexMultivariateRealFunction;
import org.apache.commons.math3.exception.ConvergenceException;
import org.apache.commons.math3.exception.TooManyIterationsException;
import org.apache.commons.math3.fitting.GaussianCurveFitter;
import org.apache.commons.math3.fitting.WeightedObservedPoint;
import org.apache.commons.math3.fitting.WeightedObservedPoints;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TooManyListenersException;

/**
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */
public class GaussianFitting
{
    /**
     * This class is used to set optimization constrains
     */
    public static class Constraints {
        Range<Double> heightRange = Range.greaterThan(0.0); // default height range = (0, inf)
        Range<Double> meanRange = Range.all();              // default mean range = (-inf, inf)
        Range<Double> sigmaRange = Range.greaterThan(0.0);  // default sigma range = (0, inf)

        public Constraints heightRange(double min, double max) {
            this.heightRange = Range.open(min, max);
            return this;
        }

        public Constraints meanRange(double min, double max) {
            this.meanRange = Range.open(min, max);
            return this;
        }

        public Constraints sigmaRange(double min, double max) {
            this.sigmaRange = Range.open(min, max);
            return this;
        }
    }

    /**
     * This class is used to keep the gaussian parameters Height, Means, and Sigma
     */

    public static class Parameters {
        public double height = 1.0;
        public double mean = 0.0;
        public double sigma = 0.0;
    }

    /**
     * This class is used to return the gaussian parameters and the fitting error
     */

    public static class Result {
        public boolean successful = false;
        public Parameters parameters = new Parameters();
        public double error = Double.MAX_VALUE;
    }

    // ----------------------------
    // ----- Global variables -----
    // ----------------------------

    private final List<WeightedObservedPoint> data;
    private Constraints constraints;

    // --------------------------
    // ----- Public Methods -----
    // --------------------------

    /**
     * Constructor
     * @param points map with the data points, the gaussian is to be fitted in.
     */

    public GaussianFitting(@Nonnull final Map<Double, Double> points)
    {
        WeightedObservedPoints observedPoints = new WeightedObservedPoints();
        for (Map.Entry<Double, Double> e : points.entrySet())
            observedPoints.add(e.getKey(), e.getValue());
        data = observedPoints.toList();
        constraints = new Constraints();
    }

    /**
     * Sets constraints for gaussian parameters height, mean, and sigma.
     * @param constraints
     */

    public void setConstraints(@Nonnull final Constraints constraints) {
        this.constraints = constraints;
    }

    /**
     * Performs gaussian fitting. The initial values for the parameters are estimated from the data points.
     * @return Result with fitted parameters and fitting error.
     */

    @Nonnull
    public Result execute()
    {
        Parameters parameters;

        try {
            parameters = getParameters(GaussianCurveFitter
                    .create()
                    .withMaxIterations(100)
                    .fit(data));
        }
        catch (Exception e) {
            return new Result();
        }

        if (!check(parameters))
            return new Result();

        Result result = new Result();
        result.successful = true;
        result.parameters = parameters;
        result.error = getError(parameters);

        return result;
    }

    /**
     * Estimates height, mean, and sigma based on the data points
     * @return estimated parameters
     */

    public Parameters estimate() {
        return getParameters(new GaussianCurveFitter.ParameterGuesser(data).guess());
    }

    private boolean check(Parameters parameters)
    {
        boolean check = constraints.heightRange.contains(parameters.height);
        check &= constraints.meanRange.contains(parameters.mean);
        check &= constraints.sigmaRange.contains(parameters.sigma);
        return check;
    }

    private double getError(Parameters p) {
        double sum = 0.0;
        for (WeightedObservedPoint point : data) {
            double x = point.getX();
            double y = point.getY();
            double delta = y - gaussian(x, p);
            sum += delta * delta;
        }
        return sum / data.size();
    }

    private Parameters getParameters(double[] x) {
        Parameters p = new Parameters();
        p.height = x[0];
        p.mean = x[1];
        p.sigma = x[2];
        return p;
    }

    public static double gaussian(double x, @Nonnull Parameters p) {
        return p.height * Math.exp(-(x - p.mean) * (x - p.mean) / (2 * p.sigma * p.sigma));
    }

}
