package dulab.adap.common.algorithms.statistics;

import org.apache.commons.math3.exception.NumberIsTooSmallException;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.NavigableMap;

/**
 * This class is designed to estimate parameters of Bi-Gaussian and finds its values
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */
public class BiGaussian
{
    public static final double FWHM_OVER_STD = 2 * Math.sqrt(2 * Math.log(2.0));

    /**
     * This class is designed to store coordinates of a point
     */
    public static class Point {
        public final double x;
        public final double y;

        public Point(double x, double y) {
            this.x = x;
            this.y = y;
        }
    }

    /**
     * This class is designed to store the parameters of bi-gaussian
     */
    public static class Estimate {
        public final double height;
        public final double mean;
        public final double leftSTD;
        public final double rightSTD;

        public Estimate(double h, double m, double ls, double rs) {
            if (h <= 0.0 || ls <= 0.0 || rs <= 0.0)
                throw new IllegalArgumentException("BiGaussian.Estimate with negative parameters");

            height = h;
            mean = m;
            leftSTD = ls;
            rightSTD = rs;
        }

        public double getFWHM() {
            double leftFWHM = leftSTD * FWHM_OVER_STD / 2;
            double rightFWHM = rightSTD * FWHM_OVER_STD / 2;
            return leftFWHM + rightFWHM;
        }
    }

    /**
     * Assuming that data has a bi-gaussian shape, finds the height, mean, and two standard deviations of the gaussians
     * @param data array of data points
     * @return array of 4 numbers: height, mean, and two standard deviations
     */
    @Nonnull
    public static Estimate estimate(@Nonnull Point[] data) throws NumberIsTooSmallException
    {
        final int size = data.length;

        if (size < 3)
            throw new NumberIsTooSmallException(data.length, 3, false);

        int maxIndex = 0;
        double maxValue = -Double.MAX_VALUE;
        for (int i = 0; i < data.length; ++i)
            if (data[i].y >= maxValue) {
                maxValue = data[i].y;
                maxIndex = i;
            }

        double halfMaxValue = maxValue / 2;

        // Find left coordinate
        double left = data[0].x;
        for (int i = maxIndex - 1; i >= 0; --i)
            if (data[i].y < halfMaxValue) {
                left = data[i].x + (data[i + 1].x - data[i].x) / (data[i + 1].y - data[i].y) * (halfMaxValue - data[i].y);
                break;
            }

        // Find right coordinate
        double right = data[size - 1].x;
        for (int i = maxIndex + 1; i < data.length; ++i)
            if (data[i].y < halfMaxValue) {
                right = data[i - 1].x + (data[i].x - data[i - 1].x) / (data[i].y - data[i - 1].y) * (halfMaxValue - data[i - 1].y);
                break;
            }

        double leftFWHM = 2 * (data[maxIndex].x - left);
        double rightFWHM = 2 * (right - data[maxIndex].x);

        return new Estimate(maxValue, data[maxIndex].x,
                leftFWHM / FWHM_OVER_STD,
                rightFWHM / FWHM_OVER_STD);
    }

    /**
     * Assuming that data has a gaussian shape, finds the height, mean, and standard deviation of the gaussian.
     * The mean and standard deviation values are calculated in terms of indices of the data array.
     * @param data array of numbers
     * @return array of 3 numbers: height, mean, and standard deviation
     */
    public static Estimate estimate(@Nonnull double[] data)
            throws NumberIsTooSmallException
    {
        Point[] points = new Point[data.length];
        for (int i = 0; i < data.length; ++i)
            points[i] = new Point(i, data[i]);

        return estimate(points);
    }

    /**
     * Evaluates bi-gaussian at point x
     * @param x argument of the bi-gaussian
     * @param e parameters of the bi-gaussian
     * @return value of the bi-gaussian
     */
    public static double biGaussian(double x, @Nonnull Estimate e) {
        double value;
        double d = x - e.mean;
        if (d > 0.0) // x > e.mean
            value = e.height * Math.exp(-d * d / (2 * e.rightSTD * e.rightSTD));
        else // x <= e.mean
            value = e.height * Math.exp(-d * d / (2 * e.leftSTD * e.leftSTD));
        return value;
    }

    /**
     * Evaluates bi-gaussian at point x
     * @param x argument of the bi-gaussian
     * @param height height of the bi-gaussian
     * @param mean mean of the bi-gaussian
     * @param leftSTD left standard deviation of the bi-gaussian
     * @param rightSTD right stnadard deviation of the bi-gaussian
     * @return value of the bi-gaussian at point x
     */
    public static double biGaussian(double x, double height, double mean, double leftSTD, double rightSTD) {
        return biGaussian(x, new Estimate(height, mean, leftSTD, rightSTD));
    }
}
