package dulab.adap.common.algorithms.statistics;

import org.apache.commons.math3.exception.NumberIsTooSmallException;

import javax.annotation.Nonnull;

/**
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */
public class Gaussian
{
    public static final double FWHM_OVER_STD = 2 * Math.sqrt(2 * Math.log(2.0));

    /**
     * This class is used to keep coordinates of a point
     */
    public static class Point {
        public final double x;
        public final double y;

        public Point(double x, double y) {
            this.x = x;
            this.y = y;
        }
    }

    public static class Estimate
    {
        public final double height;
        public final double mean;
        public final double std;

        private Estimate(double h, double m, double s) {
            height = h;
            mean = m;
            std = s;
        }
    }

    /**
     * Assuming that data has a gaussian shape, finds the height, mean, and standard deviation of the gaussian
     * @param data array of data points
     * @return array of 3 numbers: height, mean, and standard deviation
     */
    public static Estimate estimate(@Nonnull Point[] data) throws NumberIsTooSmallException
    {
        final int size = data.length;

        if (size < 3)
            throw new NumberIsTooSmallException(data.length, 3, false);

//        final String errorMessage = "Cannot estimate Gaussian parameters";

        int maxIndex = 0;
        double maxValue = -Double.MAX_VALUE;
        for (int i = 0; i < data.length; ++i)
            if (data[i].y >= maxValue) {
                maxValue = data[i].y;
                maxIndex = i;
            }

//        if (maxIndex == null)
//            throw new IllegalArgumentException(errorMessage);

        double halfMaxValue = maxValue / 2;

        // Find left coordinate
        double left = data[0].x;
        for (int i = maxIndex - 1; i >= 0; --i)
            if (data[i].y < halfMaxValue) {
                left = data[i].x + (data[i + 1].x - data[i].x) / (data[i + 1].y - data[i].y) * (halfMaxValue - data[i].y);
                break;
            }

//        if (left == null)
//            throw new NumberIsTooSmallException(maxIndex - 1, 2, false);


        // Find right coordinate
        double right = data[size - 1].x;
        for (int i = maxIndex + 1; i < data.length; ++i)
            if (data[i].y < halfMaxValue) {
                right = data[i - 1].x + (data[i].x - data[i - 1].x) / (data[i].y - data[i - 1].y) * (halfMaxValue - data[i - 1].y);
                break;
            }

//        if (right == null)
//            throw new NumberIsTooSmallException(data.length - maxIndex - 1, 2, false);

        double fwhm = right - left;

        return new Estimate(maxValue, data[maxIndex].x, fwhm / FWHM_OVER_STD);
    }

    /**
     * Assuming that data has a gaussian shape, finds the height, mean, and standard deviation of the gaussian.
     * The mean and standard deviation values are calculated in terms of indices of the data array.
     * @param data array of numbers
     * @return array of 3 numbers: height, mean, and standard deviation
     */
    public static Estimate estimate(@Nonnull double[] data) {
        Point[] points = new Point[data.length];
        for (int i = 0; i < data.length; ++i)
            points[i] = new Point(i, data[i]);

        return estimate(points);
    }

    /**
     * Evaluates gaussian at point x
     * @param x argument of the gaussian
     * @param e parameters of the gaussian
     * @return value of the gaussian
     */
    public static double gaussian(double x, @Nonnull Estimate e)
    {
        double d = x - e.mean;
        return e.height * Math.exp(-d * d / (2 * e.std * e.std));
    }
}
