/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.distances;

import dulab.adap.common.types.SparseVector;

/**
 *
 * @author aleksandrsmirnov
 */
public abstract class Distance
{
    // Distance type identifiers
    static public final int EUCLIDEAN_DISTANCE = 0;
    static public final int DOT_PRODUCT_DISTANCE = 1;
    static public final int WEIGHTED_DOT_PRODUCT_DISTANCE = 2;
    
    public abstract double call(SparseVector v1, SparseVector v2);
    
    public abstract double call(SparseVector v1, SparseVector v2, double tolerance);
    
    /**
     * Create a Distance-object corresponding to the distanceID
     * 
     * @param distanceID distance identifier
     * 
     * @return object of a class implementing Distance
     */
    
    public static Distance loadInstance(final int distanceID)
    {
        Distance distance = null;
        
        switch (distanceID) {
            case EUCLIDEAN_DISTANCE:
                distance = new EuclideanDistance();
                break;
            case DOT_PRODUCT_DISTANCE:
                distance = new DotProductDistance();
                break;
            case WEIGHTED_DOT_PRODUCT_DISTANCE:
                distance = new WeightedDotProductDistance();
        }
        
        if (distance == null)
            throw new IllegalArgumentException("There is no distance with ID = " 
                    + distanceID);
        
        return distance;
    }
    
    /**
     * Get Identifier corresponding to the passed class
     * 
     * @param distance class implementing Distance
     * 
     * @return distance identifier
     */
    
    public static int getDistanceID(final Distance distance)
    {
        Integer id = null;
        
        if (distance instanceof EuclideanDistance)
            id = EUCLIDEAN_DISTANCE;
        
        else if (distance instanceof DotProductDistance)
            id = DOT_PRODUCT_DISTANCE;
        
        else if (distance instanceof WeightedDotProductDistance)
            id = WEIGHTED_DOT_PRODUCT_DISTANCE;
        
        if (id == null)
            throw new IllegalArgumentException("Couldn't find distance ID");
        
        return id;
    }
}
