/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.parsers;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 *
 * @author aleksandrsmirnov
 */
public class JCAMPParser extends Parser
{   
    private static final String NAME_FIELD = "CAS NAME";
    private static final String FORMULA_FIELD = "MOLFORM";
    private static final String CASNO_FIELD = "CAS REGISTRY NO";
    
    private final List <CompoundInfo> info;
    private final List <Map <Double, Double>> data;
    
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    public JCAMPParser(String fileName) {
        this(fileName, 1024);
    }
    
    public JCAMPParser(String fileName, int expectedSize)
    {   
        this.info = new ArrayList <> (expectedSize);
        this.data = new ArrayList <> (expectedSize);
        
        try (BufferedReader buffer = new BufferedReader(new FileReader(fileName)))
        {   
            String key = null, value = null;
            Map <String, String> compoundInfo = new HashMap <> ();
            Map <Double, Double> compoundData = new HashMap <> ();
            
            String line = buffer.readLine();
            
            while(line != null) 
            {
                if (line.length() >= 2 && line.substring(0, 2).equals("##")) // new field
                {
                    // Save previous key and value
                    if (key != null) compoundInfo.put(key, value);

                    // Start new key and value
                    String[] pair = line.substring(2).split("=", 2);
                    if (pair.length != 2) 
                        throw new IllegalArgumentException("Couldn't read line \"" +
                            line + "\"");

                    key = pair[0].trim();
                    value = pair[1].trim();

                    if (key.equals("XYDATA"))
                    {
                        compoundInfo.put(key, value);
                        
                        // Read data
                        line = readData(buffer.readLine(), buffer, compoundData);
                        
                        // Save info and data
                        if (!compoundInfo.isEmpty() && !compoundData.isEmpty()) // Succesfull attempt
                        {
                            this.info.add(processInfo(compoundInfo));
                            this.data.add(new HashMap <> (compoundData));
                        }
                        
                        // Clear compound info
                        compoundInfo.clear();
                        compoundData.clear();
                        key = null; 
                        value = null;
                        
                        //System.out.println(line);
                        
                        continue;
                    }

                } 
                else { // add the whole string to value
                    value += " " + line.trim();
                }
                
                line = buffer.readLine();
            }
        } catch (IOException e) {
            throw new IllegalArgumentException("Couldn't find file " + fileName);
        }
            
    }
    
    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    private String readData(String line, BufferedReader buffer, 
            Map <Double, Double> data) 
            throws IOException
    {
        String[] pair;
        
        while (line != null) 
        {
            pair = line.trim().split(" ", 2);
            
            try {
                data.put(
                        Double.parseDouble(pair[0]),
                        Double.parseDouble(pair[1]));
            } catch (NumberFormatException 
                    | NullPointerException 
                    | ArrayIndexOutOfBoundsException e) {
                break;
            }
            
            line = buffer.readLine();
        }
        
        return line;
    }
    
    private CompoundInfo processInfo(Map <String, String> info) 
    {
        CompoundInfo result = new CompoundInfo();
        
        for (Map.Entry <String, String> entry : info.entrySet()) 
        {
            switch (entry.getKey()) {
                case NAME_FIELD:
                    result.name = entry.getValue(); break;
                    
                case FORMULA_FIELD:
                    result.formula = entry.getValue(); break;
                    
                case CASNO_FIELD:
                    result.ID = entry.getValue(); break;
            }
        }
        
        return result;
    }
    
    // ------------------------------------------------------------------------
    // ----- Properties -------------------------------------------------------
    // ------------------------------------------------------------------------
    
    @Override
    public List <CompoundInfo> info() {return this.info;}
    
    @Override
    public List <Map <Double, Double>> data() {return this.data;}
}
