/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.parsers;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 *
 * @author aleksandrsmirnov
 */
public class MSPParser extends Parser
{   
    private static final String NAME_FIELD = "Name";
    private static final String FORMULA_FIELD = "Formula";
    private static final String ID_FIELD = "DB#";
    private static final String CASNO_FIELD = "CAS#";
    private static final String NISTNO_FIELD = "NIST#";
    
    private List <CompoundInfo> info = null;
    private List <Map <Double, Double>> data = null;
    
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    public MSPParser(String fileName)
    {
        String line = "";
        
        try (BufferedReader reader = new BufferedReader(new FileReader(fileName))) {
            //this.reader = new BufferedReader(new FileReader(fileName));
            
            this.info = new ArrayList <> ();
            this.data = new ArrayList <> ();
            
            Map <String, String> compoundInfo = new HashMap <> ();
            Map <Double, Double> compoundData = new HashMap <> ();    
            
            while ((line = reader.readLine()) != null)
            {
                // Save the current Info and Data if an empty line is met
                if (line.trim().isEmpty()) {
                    if (compoundInfo.size() > 0 && compoundData.size() > 0) {
                        this.info.add(processInfo(compoundInfo));
                        compoundInfo.clear();
                        
                        this.data.add(new HashMap(compoundData));
                        compoundData.clear();
                    }
                }
                // If the line contains the colon, then add it to Info
                else if (line.contains(":")) 
                {
                    for (String entry : line.split(";")) {
                        String[] pair = entry.trim().split(":", 2);
                        if (pair.length == 2)
                            compoundInfo.put(pair[0].trim(), pair[1].trim());
                    }
                }
                // Otherwise, assume that the line contains data
                else {
                    for (String entry : line.split(";")) {
                        String[] pair = entry.trim().split(" ", 2);
                        if (pair.length == 2) {
                            try {
                                compoundData.put(
                                        Double.parseDouble(pair[0].trim()), 
                                        Double.parseDouble(pair[1].trim()));
                            } catch (NumberFormatException 
                                    | NullPointerException e) 
                            {
                                System.out.println("Couldn't read data from '" 
                                        + entry + "': " + e.getMessage());
                            }
                        }
                    }
                }
            }
            
            // Save the last Info and Data
            if (compoundInfo.size() > 0 && compoundData.size() > 0) {
                this.info.add(processInfo(compoundInfo));
                this.data.add(new HashMap(compoundData));
            }
            
        } catch (FileNotFoundException e) {
            System.out.println("Couldn't read file '" + fileName + "'");
        } catch (IOException e) {
            System.out.println("I/O Error after line '" + line + "': " +
                    e.getMessage());
        }
    }
    
    private CompoundInfo processInfo(Map <String, String> info) 
    {
        CompoundInfo result = new CompoundInfo();
        
        for (Entry <String, String> entry : info.entrySet()) 
        {
            switch (entry.getKey()) {
                case NAME_FIELD:
                    result.name = entry.getValue(); break;
                    
                case FORMULA_FIELD:
                    result.formula = entry.getValue(); break;
                    
                case ID_FIELD:
                    if (result.ID == null)
                        result.ID = "ID: " + entry.getValue() + ";";
                    else
                        result.ID += " ID: " + entry.getValue() + ";";
                    break;
                    
                case CASNO_FIELD:
                    if (result.ID == null)
                        result.ID = "CASNO: " + entry.getValue() + ";";
                    else
                        result.ID += " CASNO: " + entry.getValue() + ";";
                    break;
                    
                case NISTNO_FIELD:
                    if (result.ID == null)
                        result.ID = "NISTNO: " + entry.getValue() + ";";
                    else
                        result.ID += " NISTNO: " + entry.getValue() + ";";
                    break;
            }
        }
        
        return result;
    }
    
    // ------------------------------------------------------------------------
    // ----- Properties -------------------------------------------------------
    // ------------------------------------------------------------------------
    
    @Override
    public List <CompoundInfo> info() {return this.info;}
    
    @Override
    public List <Map <Double, Double>> data() {return this.data;}
}
