/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.types;


import dulab.adap.common.distances.Distance;
import dulab.adap.common.parsers.CompoundInfo;

import java.io.File;
import java.nio.file.FileAlreadyExistsException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

class BallNodeInfo {
    public int id;
    public Integer leftID;
    public Integer rightID;
    public double radius;
}
/**
 *
 * @author aleksandrsmirnov
 */
public class BallTreeSQLiteInterface 
{
    private static final String NAME_FIELD = "Name";
    private static final String FORMULA_FIELD = "Formula";
    private static final String ID_FIELD = "ID";
    
    // ------------------------------------------------------------------------
    // ----- Save -------------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public static void saveBallTree(
            final String fileName, final BallTree tree)
            throws FileAlreadyExistsException
    {
        File file = new File(fileName);
        if (file.exists())
            throw new FileAlreadyExistsException("File already exists: " + fileName);
        
        Connection c;
        Statement s;
        
        try {
            //Class.forName("org.sqlite.JDBC");
            c = DriverManager.getConnection("jdbc:sqlite:" + file.getAbsolutePath());
            c.setAutoCommit(false);
            
            s = c.createStatement();
            
            // Create table METADATA
            
            String sql = "CREATE TABLE METADATA " +
                    "(PROPERTY_NAME     TEXT    NOT NULL," +
                    " PROPERTY_VALUE    INT     NOT NULL)";
            s.executeUpdate(sql);
            
            // Create table INFO
        
            /*sql = "CREATE TABLE INFO " +
                    "(ID                INT     NOT NULL," +
                    " PROPERTY_NAME     TEXT    NOT NULL," +
                    " PROPERTY_VALUE    TEXT)";*/
            sql = "CREATE TABLE INFO " +
                    "(ID        INT     NOT NULL, " + 
                    " NAME      TEXT, " +
                    " FORMULA   VARCHAR)";
            s.executeUpdate(sql);

            // Create table DATA

            sql = "CREATE TABLE DATA " +
                    "(ID                INT     NOT NULL," +
                    " MZ_VALUE          REAL    NOT NULL," +
                    " INTENSITY         REAL    NOT NULL)";
            s.executeUpdate(sql);
            
            // Create table TREE
            
            sql = "CREATE TABLE TREE " +
                    "(ID                INT     NOT NULL," +
                    " PARENT_ID         INT," + 
                    " LEFT_ID           INT," +
                    " RIGHT_ID          INT," +
                    " RADIUS            REAL    NOT NULL)";
            s.executeUpdate(sql);
            
            saveMetaData(s, tree);
            c.commit();
            
            saveData(s, tree.data());
            c.commit();
            
            saveNode(s, tree.root());
            c.commit();
            
            s.close();
            c.close();
            
        } catch (SQLException e) {
            e.printStackTrace();
        }
    }
    
    private static void saveMetaData(
            final Statement s, final BallTree tree)
            throws SQLException
    {
        s.executeUpdate("INSERT INTO METADATA (PROPERTY_NAME, PROPERTY_VALUE) " +
                "VALUES ('SIZE', " + tree.data().nrows() + ")");
        
        s.executeUpdate("INSERT INTO METADATA (PROPERTY_NAME, PROPERTY_VALUE) " +
                "VALUES ('DISTANCE_ID', " 
                + Distance.getDistanceID(tree.distanceFunction()) 
                + ")");
    }
    
    private static void saveData(final Statement s, final SparseMatrix data) 
                throws SQLException
    {
        String sql;
        
        long id = 0;
                
        for (final SparseVector vector : data.rows())
        {
            CompoundInfo info = vector.getInfo();
            
            /*sql = "INSERT INTO INFO (ID, PROPERTY_NAME, PROPERTY_VALUE) " +
                        "VALUES (" + id + ", '" + NAME_FIELD + "', '" + info.name.replaceAll("'", "") + "')";
            s.executeUpdate(sql);
            
            sql = "INSERT INTO INFO (ID, PROPERTY_NAME, PROPERTY_VALUE) " +
                        "VALUES (" + id + ", '" + FORMULA_FIELD + "', '" + info.formula.replaceAll("'", "") + "')";
            s.executeUpdate(sql);
            
            sql = "INSERT INTO INFO (ID, PROPERTY_NAME, PROPERTY_VALUE) " +
                        "VALUES (" + id + ", '" + ID_FIELD + "', '" + info.ID.replaceAll("'", "") + "')";
            s.executeUpdate(sql);*/
            
            sql = "INSERT INTO INFO (ID, NAME, FORMULA) " +
                    "VALUES (" + id + ", '" + info.name.replaceAll("'", "") + 
                    "', '" + info.formula.replaceAll("'", "") + "')";
            s.execute(sql);
            
            for (Entry <Double, Double> e : vector.data().entrySet()) {
                sql = "INSERT INTO DATA (ID, MZ_VALUE, INTENSITY) " +
                        "VALUES (" + id + ", " + e.getKey() + ", " + e.getValue() + ")";
                s.executeUpdate(sql);
            }
            
            ++id;
        }
    }
        
    private static void saveNode(final Statement s, final BallNode node)
                throws SQLException
    {
        if (node == null) return;
        
        String sql = "INSERT INTO TREE (ID, PARENT_ID, LEFT_ID, RIGHT_ID, RADIUS) " +
                "VALUES (" + node.getID() + ", ";
        
        if (node.parent != null)
            sql += node.parent.getID() + ", ";
        else
            sql += "NULL, ";
        
        if (node.left != null)
            sql += node.left.getID() + ", ";
        else 
            sql += "NULL, ";
        
        if (node.right != null)
            sql += node.right.getID() + ", ";
        else
            sql += "NULL, ";
        
        sql += node.getRadius() + ")";
        
        s.executeUpdate(sql);
        
        saveNode(s, node.left);
        saveNode(s, node.right);
    }
        
    // ------------------------------------------------------------------------
    // ----- Load -------------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public static BallTree loadBallTree(final String fileName)
            throws Exception
    {
        BallTree result = null;
        
        Connection c;
        Statement s;
        
        try {
            //Class.forName("org.sqlite.JDBC");
            c = DriverManager.getConnection("jdbc:sqlite:" + fileName);
            c.setAutoCommit(false);
            
            s = c.createStatement();
            
            Map <String, Integer> meta = loadMetaData(s);
            Integer size = meta.get("SIZE");
            Distance distance = Distance.loadInstance(meta.get("DISTANCE_ID"));
            
            List <SparseVector> data = loadData(s, size);
            List <CompoundInfo> info = loadInfo(s, size);
            
            if (data.size() != info.size())
                throw new Exception("Corrupted data");
            
            size = data.size();
            
            for (int i = 0; i < size; ++i)
                data.get(i).setInfo(info.get(i));
            
            BallNode root = loadRoot(s, size);
            
            result = new BallTree(new SparseMatrix(data), distance, root);
            
            s.close();
            c.close();
            
        } catch (SQLException e) {
            e.printStackTrace();
        }
        
        return result;
    }
    
    private static Map <String, Integer> loadMetaData
        (final Statement s) throws SQLException
    {
        Map <String, Integer> result = new HashMap <> ();
                
        ResultSet rs = s.executeQuery("SELECT * FROM METADATA");
        
        while(rs.next())
        {
            String propertyName = rs.getString("PROPERTY_NAME");
            Integer propertyValue = rs.getInt("PROPERTY_VALUE");
            result.put(propertyName, propertyValue);
        }
        
        return result;
    }
    
    private static List <SparseVector> loadData
        (final Statement s, final Integer size) throws SQLException
    {
        List <SparseVector> result;
        
        if (size != null)
            result = new ArrayList <> (size);
        else
            result = new ArrayList <> ();
        
        Map <Double, Double> spectrum = new HashMap <> ();
        
        ResultSet rs = s.executeQuery("SELECT * FROM DATA");
            
        long prevIndex = 0;
        while(rs.next())
        {
            long index = rs.getLong("ID");
            double mzValue = rs.getDouble("MZ_VALUE");
            double intensity = rs.getDouble("INTENSITY");

            if (index != prevIndex) {
                result.add(new SparseVector(new TreeMap <> (spectrum)));
                spectrum.clear();
                prevIndex = index;
            }

            spectrum.put(mzValue, intensity);
        }

        result.add(new SparseVector(new TreeMap <> (spectrum)));

        rs.close();
        
        return result;
    }
    
    private static List <CompoundInfo> loadInfo
        (final Statement s, final Integer size) throws SQLException
    {
        List <CompoundInfo> result;
        
        if (size != null)
            result = new ArrayList <> (size);
        else
            result = new ArrayList <> ();
        
        /*Map <String, String> info = new HashMap <> ();
        
        ResultSet rs = s.executeQuery("SELECT * FROM INFO");
        
        long prevIndex = 0;
        while(rs.next())
        {
            int index = rs.getInt("ID");
            String propertyName = rs.getString("PROPERTY_NAME");
            String propertyValue = rs.getString("PROPERTY_VALUE");
            
            if (index != prevIndex) {
                result.add(new CompoundInfo()
                        .name(info.get(NAME_FIELD))
                        .formula(info.get(FORMULA_FIELD))
                        .id(info.get(ID_FIELD)));
                info.clear();
                prevIndex = index;
            }
            
            info.put(propertyName, propertyValue);
        }
        result.add(new CompoundInfo()
                .name(info.get(NAME_FIELD))
                .formula(info.get(FORMULA_FIELD))
                .id(info.get(ID_FIELD)));
        */
        
        ResultSet rs = s.executeQuery("SELECT * FROM INFO");
        while (rs.next())
        {
            result.add(new CompoundInfo()
                .name(rs.getString("NAME"))
                .formula(rs.getString("FORMULA"))
                .id(Integer.toString(rs.getInt("ID"))));
        }
        
        rs.close();
        
        return result;
    }
    
    private static BallNode loadRoot(final Statement s, int size) 
            throws SQLException 
    {
        // Read all rows
        ResultSet rs = s.executeQuery("SELECT * FROM TREE ORDER BY ID");
        
        List <BallNodeInfo> info = new ArrayList <> (size);
        
        while(rs.next()) 
        {
            BallNodeInfo nodeInfo = new BallNodeInfo();
            
            nodeInfo.id = rs.getInt("ID");
            nodeInfo.radius = rs.getDouble("Radius");
            
            nodeInfo.leftID = rs.getInt("LEFT_ID");
            if (rs.wasNull()) nodeInfo.leftID = null;
            
            nodeInfo.rightID = rs.getInt("RIGHT_ID");
            if (rs.wasNull()) nodeInfo.rightID = null;
            
            info.add(nodeInfo);
        }
        
        rs.close();
        
        // Read the first row
        
        rs = s.executeQuery("SELECT * FROM TREE LIMIT 1");
        
        if (!rs.next()) return null;
        
        double radius = rs.getDouble("RADIUS");
        int id = rs.getInt("ID");

        int leftID = rs.getInt("LEFT_ID");
        boolean goLeft = !rs.wasNull();

        int rightID = rs.getInt("RIGHT_ID");
        boolean goRight = !rs.wasNull();

        rs.close();
        
        // Build a tree
        
        BallNode root = new BallNode(id, null);
        root.setRadius(radius);
        if (goLeft) root.left = loadNode(info, leftID, root);
        if (goRight) root.right = loadNode(info, rightID, root);
        
        return root;
    }
        
    private static BallNode loadNode(List <BallNodeInfo> info, int index,
            final BallNode parent) throws SQLException
    {
        BallNodeInfo nodeInfo = info.get(index);
        
        if (nodeInfo == null) return null; // Add 1 because SQL row numbers start from 1
        
        BallNode node = new BallNode(index, parent);
        node.setRadius(nodeInfo.radius);

        if (nodeInfo.leftID != null)
            node.left = loadNode(info, nodeInfo.leftID, node);
        
        if (nodeInfo.rightID != null)
            node.right = loadNode(info, nodeInfo.rightID, node);
        
        return node;
    }
    
    /*
    private static BallNode loadNode(final Statement s)
            throws SQLException
    {
        // Read the first row
        ResultSet rs = s.executeQuery("SELECT * FROM TREE LIMIT 1");
        
        BallNode node = null;
        
        while (rs.next()) 
        {
            int id = rs.getInt("ID");
            node = new BallNode(id, null);
            
            double radius = rs.getDouble("RADIUS");
            node.setRadius(radius);
            
            int leftID = rs.getInt("LEFT_ID");
            if (rs.wasNull())
            
            int rightID = rs.getInt("RIGHT_ID");
            
            
            node.setRadius(radius);
            node.left = loadNode(s, leftID, node);
            node.right = loadNode(s, rightID, node);
        }
        
        rs.close();
    }*/
}
