package dulab.adap.common.types;

import com.google.common.collect.Range;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

/**
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */
public class BinarySearchTree<T extends BinarySearchTree.Item>
{
    /**
     * Interface for any object with a double value
     */
    public interface Item {
        double getValue();
    }

    /**
     * This class represents a node of the binary search tree
     */
    private class Node
    {
        final T item;
        final Range<Double> range;
        public Node left;
        public Node right;

        Node(@Nonnull T item, double start, double end) {
            this.item = item;
            this.range = Range.closed(start, end);
            this.left = null;
            this.right = null;
        }
    }

    /** Root of the binary search tree */
    private final Node root;

    /**
     * Creates an instance of the binary search tree
     * @param items array of items of the binary search tree
     */
    public BinarySearchTree(@Nonnull T[] items)
    {
        Arrays.sort(items, new Comparator<T>() {
            @Override
            public int compare(Item o1, Item o2) {
                return Double.compare(o1.getValue(), o2.getValue());
            }
        });

        root = buildTree(items);
    }

    /**
     * Builds the binary search tree. For a given array of items:
     *   > Find the median item
     *   > Create a node with the median item
     *   > Recursively build a tree for the left and right halves of the array
     *
     * @param items array of items, the tree is built for
     * @return the root of the binary search tree
     */
    private Node buildTree(@Nonnull T[] items)
    {
        final int numItems = items.length;

        if (numItems == 0) return null;

        final Node node = new Node(
                items[numItems / 2],
                items[0].getValue(),
                items[numItems - 1].getValue());

        node.left = buildTree(Arrays.copyOfRange(items, 0, numItems / 2));
        node.right = buildTree(Arrays.copyOfRange(items, numItems / 2 + 1, numItems));

        return node;
    }

    /**
     * Finds all items with their values within the specified range
     * @param range range of values, we search for
     * @return list of items within the specified range
     */
    @Nonnull
    public List<T> search(@Nonnull Range<Double> range)
    {
        List<T> searchResult = new ArrayList<>();

        searchStep(range, root, searchResult);

        return searchResult;
    }

    /**
     * Recursive search of items within a specified range. For a given node:
     *   > If value of the node lies in the range, add it to the search result
     *   > If left branch is not null and its range overlaps with the search range, descent to the left branch
     *   > If right branch is not null and its range overlaps with the search range, descent to the right branch
     *
     * @param searchRange ranges of values, we search for
     * @param node current node of the binary search tree
     * @param searchResult list of items within the search range
     */
    private void searchStep(@Nonnull Range<Double> searchRange, @Nonnull Node node, @Nonnull List<T> searchResult)
    {
        if (searchRange.contains(node.item.getValue()))
            searchResult.add(node.item);

        if (node.left != null && node.left.range.isConnected(searchRange))
            searchStep(searchRange, node.left, searchResult);

        if (node.right != null && node.right.range.isConnected(searchRange))
            searchStep(searchRange, node.right, searchResult);
    }
}
