/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.common.types;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 *
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */


public class Graph <T>
{
    class Edge
    {
        private final T startNode;
        private final T endNode;
        
        public Edge(T start, T end) {
            startNode = start;
            endNode = end;
        }
    }
    
    Set <T> nodes;
    List <Edge> edges;
    
    public Graph()
    {
        this.nodes = new HashSet <> ();
        this.edges = new ArrayList <> ();
    }
    
    public Graph(T[] nodes, double[][] similarity, double threshold)
    {
        final int size = similarity.length;
        
        this.nodes = new HashSet <> ();
        this.edges = new ArrayList <> ();
        
        for (int i = 0; i < size; ++i)
        {
            this.nodes.add(nodes[i]);
            
            for (int j = i + 1; j < size; ++j)
                if (similarity[i][j] > threshold)
                    this.edges.add(new Edge(nodes[i], nodes[j]));
        }
    }
    
    public void addNode(T node) {this.nodes.add(node);}
    
    public void addEdge(T startNode, T endNode) 
    {
        this.nodes.add(startNode);
        this.nodes.add(endNode);
        
        if (startNode.equals(endNode)) return;
        
        this.edges.add(new Edge(startNode, endNode));
    }
    
    public int numNodes() {return this.nodes.size();}
    
    public int numEdges() {return this.edges.size();}
    
    public Set <T> nodes() {return this.nodes;}
    
    /**
     * Degree of a node (number of edges connected to the node)
     * @param node a node
     * @return number of edges
     */
    
    public int degree(T node) 
    {
        int degree = 0;
        
        for (Edge e : this.edges)
            if (e.startNode.equals(node) || e.endNode.equals(node)) ++degree;
        
        return degree;
    }
    
    /**
     * Number of edges that have one or both vertices from the set
     * @param nodes set of nodes
     * @return number of edges
     */
    
    public int numAdjasentEdges(Set <T> nodes)
    {
        int count = 0;
        
        for (Edge e : this.edges)
            if (nodes.contains(e.startNode) || nodes.contains(e.endNode)) 
                ++count;
        
        return count;
    }
    
    /**
     * Find modularity of partition of a graph into communities
     * 
     * Q = 1 / (2 L) * Sum(kappa[i] - k[i] a[i])
     * 
     *     L        = total number of edges
     *     kappa[i] = number of edges that a node i has with other nodes in its 
     *                community
     *     k[i]     = degree of node i
     *     a[i]     = fraction of edges that have one or both vertices inside
     *                the community
     * 
     * @param communities communities (subgraphs of the graph)
     * @return modularity
     */
    
    public double modularity(final List <Graph <T>> communities)
    {
        int totalNumEdges = this.numEdges();
        
        double sum = 0.0;
        
        for (Graph <T> community : communities)
        {
            double a = this.numAdjasentEdges(community.nodes()) / totalNumEdges;
            
            for (T node : community.nodes())
            {
                int kappa = community.degree(node);
                int k = this.degree(node);
                
                sum += kappa - k * a;
            }
        }
        
        sum /= 2 * totalNumEdges;
        
        return sum;
    }
}
