/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.common.types;

import dulab.adap.common.parsers.CompoundInfo;
import java.io.Serializable;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableMap;
import java.util.TreeMap;

/**
 *
 * @author aleksandrsmirnov
 */
public class SparseVector implements Serializable
{
    private static final double DEFAULT_VALUE = 0.0;
    
    private final NavigableMap <Double, Double> data; // We keep vector as pairs (index, value)
    private CompoundInfo info;
    //private int size;
    
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    public SparseVector() {
        this.data = new TreeMap <> ();
        this.info = null;
        //this.size = 0;
    }
    
    /*
    public SparseVector(final int size) {
        this.data = new TreeMap <> (size);
        this.info = null;
    }
*/
    
    public SparseVector(final SparseVector other) {
        this(other.data);
    }
    
    public SparseVector(final NavigableMap <Double, Double> data) {
        //this.data = new HashMap <> (data);
        this.data = data;
        this.info = null;
        
        //double maxIndex = 0;
        //for (final double index : this.data.keySet())
//            if (index > maxIndex) maxIndex = index;
        
        //this.size = maxIndex + 1;
    }
    
    public SparseVector(final Map <Double, Double> data) {
        this(new TreeMap <> (data));
    }
    
    public SparseVector(double ... values) 
    {
        this.data = new TreeMap <> ();
        
        for (int i = 0; i < values.length; ++i)
            if (values[i] != DEFAULT_VALUE)
                this.data.put((double) i, values[i]);
        
        this.info = null;
        
        //this.size = values.length;
    }
    
    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public void put(double index, double value) {
        this.data.put(index, value);
        
        //int newSize = index + 1;
        //if (newSize > size) size = newSize;
    }
    
    public void setInfo(CompoundInfo info) {
        this.info = info;
    }
    
    /**
     * Finds the sum of two sparse vectors
     * 
     * @param other the second sparse vector
     * @return the sparse vector of the sum
     */
    
    public SparseVector add(final SparseVector other) {
        return add(other, true);
    }
    
    public SparseVector add(final SparseVector other, boolean copy) 
    {
        SparseVector result = this;
        
        if (copy) 
            result = new SparseVector(this);
        
        for (Entry <Double, Double> e : other.data.entrySet()) {
            double index = e.getKey();
            Double value =  result.data.get(index);
            if (value == null) value = DEFAULT_VALUE;

            result.data.put(index, e.getValue() + value);
            
            //int newSize = index + 1;
            //if (newSize > this.size) this.size = newSize + 1;
        }
        
        return result;
    }
    
    /**
     * Finds the difference of two sparse vectors
     * 
     * @param other the second sparse vector
     * @return the sparse vector of the difference
     */
    
    public SparseVector substract(final SparseVector other) {
        return substract(other, true);
    }
    
    public SparseVector substract(final SparseVector other, boolean copy)
    {
        SparseVector result = this;
        
        if (copy)
            result = new SparseVector(this);
        
        for (Entry <Double, Double> e : other.data.entrySet()) {
            //result.data.compute(e.getKey(), (k,v) -> v == null ? -e.getValue() : v - e.getValue());
            double index = e.getKey();
            Double value = result.data.get(index);
            if (value == null) value = DEFAULT_VALUE;
            
            result.data.put(index, - e.getValue() + value);
            
            //int newSize = index + 1;
            //if (newSize > this.size) this.size = newSize + 1;
        }
        
        return result;
    }
        
    /**
     * Multiplies a sparse vector by a scalar
     * 
     * @param x the scalar
     * @return the sparse vector of the multiplication
     */
    
    public SparseVector multiply(final double x) {
        return multiply(x, true);
    }
    
    public SparseVector multiply(final double x, boolean copy) 
    {
        SparseVector result = this;
        
        if (copy)
            result = new SparseVector(this);
        
        for (Entry <Double, Double> e : result.data.entrySet())
            e.setValue(x * e.getValue());
        
        return result;
    }
    
    public SparseVector trim(double min)
    {
        SparseVector result = new SparseVector();
        
        for (Entry <Double, Double> entry : this.data.entrySet()) {
            final double value = entry.getValue();
            if (value > min)
                result.data.put(entry.getKey(), value);
        }
            
        return result;
    }
    
    /**
     * Finds the l2-norm of a sparse vector
     * 
     * @return the norm
     */
    
    public double norm()
    {
        double sum = 0.0;
        
        for (double value : this.data.values())
            sum += value * value;
        
        return java.lang.Math.sqrt(sum);
    }
    
    // ------------------------------------------------------------------------
    // ----- Properties -------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public double get(double index) {
        Double value = this.data.get(index);
        if (value == null) value = DEFAULT_VALUE;
        return value;
    }
    
    public Map <Double, Double> data() {return this.data;}
    
    public CompoundInfo getInfo() {return this.info;}
    
    public int size() {return this.data.size();}
    
    
    @Override
    public String toString() 
    {
        String result = "[";
        
        for (int i = 0; i < this.data.size(); ++i)
            result += Double.toString(this.get(i)) + ",";
        result += "]";
        
        return result;
    }
}
