package dulab.adap.datamodel;

import dulab.adap.common.types.Function;

import javax.annotation.Nonnull;
import java.util.Arrays;

/**
 * This class is designed to store a chromatogram
 *
 * @author Du-Lab Team <dulab.binf@gmail.com>
 */
public class Chromatogram extends Function
{
    /** L2-norm of the chromatogram */
    public final double norm;

    /** Index of the highest point in the chromatogram */
    private final int apexIndex;

    /**
     * Creates an instance of the class. Arrays retTimes and intensities are required to have the same size.
     * Array retTimes has to have values in the ascending order.
     * @param retTimes retention times of the chromatogram
     * @param intensities intensities of the chromatogram
     */
    public Chromatogram(@Nonnull double[] retTimes, @Nonnull double[] intensities)
    {
        super(retTimes, intensities);

        int index = 0;
        double height = -Double.MAX_VALUE;
        double norm2 = 0.0;
        for (int i = 0; i < length; ++i)
        {
            if (intensities[i] > height) {
                height = intensities[i];
                index = i;
            }

            if (i != 0) {
                double d = retTimes[i] - retTimes[i - 1];
                if (d <= 0.0) throw new IncorrectRetTimesException();

                double h = 0.5 * (intensities[i] * intensities[i] + intensities[i - 1] * intensities[i - 1]);
                norm2 += d * h;
            }
        }
        this.apexIndex = index;
        this.norm = Math.sqrt(norm2);
    }

    /** Returns the apex retention time of the chromatogram */
    public double getApexRetTime() {return xs[apexIndex];}

    /** Returns the first retention time in the chromatogram */
    public double getFirstRetTime() {return xs[0];}

    /** Returns the last retention time in the chromtogram */
    public double getLastRetTimes() {return xs[length - 1];}

    /** Returns the apex intensity of the chromatogram */
    public double getHeight() {return ys[apexIndex];}

    /** Returns the i-th retention time in the chromatogram */
    public double getRetTime(int index) {return xs[index];}

    /** Returns the i-th intensity in the chromatogram */
    public double getIntensity(int index) {return ys[index];}

    /**
     * Returns intensity corresponding to the given retention time. If no such intensity exists, returns null.
     * @param retTime any number
     * @return intensity corresponding to the given x. If no such intensity exists, returns null.
     */
//    public Double getIntensity(double retTime) {
//        Double y = null;
//        int index = Arrays.binarySearch(xs, retTime);
//        if (0 <= index && index < length && xs[index] == retTime)
//            y = ys[index];
//        return y;
//    }
    public Double getIntensity(double retTime) {
        return getIntensity(retTime, false);
    }

    public Double getIntensity(double retTime, boolean interpolate) {
        int index = Arrays.binarySearch(xs, retTime);
        boolean contained = 0 <= index && index < length;

        if (contained)
            return ys[index];

        if (interpolate) {
            int i2 = -index - 1;
            int i1 = i2 - 1;
            if (i1 >= 0 && i2 < length)
                return ys[i1] + (retTime - xs[i1]) * (ys[i2] - ys[i1]) / (xs[i2] - xs[i1]);
            else
                return 0.0;
        }

        return null;
    }

    public double interpolateIntensity(double retTime) {
        int index = Arrays.binarySearch(xs, retTime);
        if (index >= 0)
            return ys[index];
        else {
            int insertionPoint = -index - 1;
            if (insertionPoint == 0 || insertionPoint == length)
                return 0.0;
            else {
                double x1 = xs[insertionPoint - 1];
                double x2 = xs[insertionPoint];
                double y1 = ys[insertionPoint - 1];
                double y2 = ys[insertionPoint];
                return y1 + (retTime - x1) / (x2 - x1) * (y2 - y1);
            }
        }
    }

    /**
     * Checks if retTimes contain the given retention time.
     * @param retTime any number
     * @return true if retTimes contain the given retention time, false otherwise
     */
    public boolean contains(double retTime) {
        boolean result = false;
        int index = Arrays.binarySearch(xs, retTime);
        if (0 <= index && index < length && xs[index] == retTime)
            result = true;
        return result;
    }

    // ----------------------
    // ----- Exceptions -----
    // ----------------------

    /**
     * Exception that is raised when unsorted retention times are passed to the constructor
     */
    public static class IncorrectRetTimesException extends IllegalArgumentException {
        public IncorrectRetTimesException() {
            super("Array of retention times is not sorted");
        }
    }
}
