/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.NavigableMap;
import java.util.TreeMap;

/**
 *
 * @author aleksandrsmirnov
 */
public class Component 
        implements Serializable, Comparator <Component>, Comparable <Component> 
{
    
    private final List <Peak> peaks;
    private final Peak bestPeak;
    private final NavigableMap <Double, Double> spectrum;
    private final NavigableMap <Double, Double> scale;
    
    private final double norm; // Spectrum norm
    private double mz;
    boolean isReference;
    boolean isAligned;
    double shift;
    
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    public Component(final Component component) {
        
        this(component.getPeaks(), component.getBestPeak(),
                component.getSpectrum(), component.getScale());
        
        this.isReference = component.isReference;
        this.isAligned = component.isAligned;
        this.shift = component.shift;
    }
    
    public Component(final List <Peak> peaks, final Peak bestPeak,
            final NavigableMap <Double, Double> spectrum,
            final NavigableMap <Double, Double> scale)
    {
//        if (peaks != null) {
//            this.peaks = new ArrayList <> (peaks.size());
//            for (final Peak peak : peaks) this.peaks.add(new Peak(peak));
//        } else
//            this.peaks = new ArrayList <> ();
        this.peaks = peaks;
        
        //this.bestPeak = new Peak(bestPeak);
        this.bestPeak = bestPeak;
        this.mz = bestPeak.getMZ();
        //this.spectrum = new TreeMap <> (spectrum);
        this.spectrum = spectrum;
        
        this.scale = scale;
//        if (scale != null)
//            this.scale = new TreeMap <> (scale);
//        else
//            this.scale = new TreeMap <> ();
        
        
        isReference = false;
        isAligned = false;
        shift = 0.0;
        
        norm = java.lang.Math.sqrt(dulab.adap.common.algorithms.Math.
                discrete_dot_product(spectrum, spectrum, 1e-6));
    }
    
    static public Component deserialize(final byte[] serializedComponent) {
        Component result = null;
        
        try {
            ByteArrayInputStream bis = new ByteArrayInputStream(
                    serializedComponent);
            ObjectInputStream is = new ObjectInputStream(bis);
            result = (Component) is.readObject();

            is.close();
            bis.close();
        } catch (IOException | ClassNotFoundException e) {
            e.printStackTrace();
        }
        
        return result;
    }
    
    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public void setShift(double shift) {
        this.shift = shift;
        isAligned = true;
        bestPeak.setShift(shift);
        if (peaks != null)
            for (Peak peak : peaks) peak.setShift(shift);
    }
    
    public void setAsReference() {
        isReference = true;
        isAligned = true;
    }
    
    public byte[] serialize() {
        byte[] result = null;
        
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ObjectOutputStream os = new ObjectOutputStream(bos);
        
            os.writeObject(this);
            result = bos.toByteArray();
            
            os.close();
            bos.close();
            
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        return result;
    }
    
    @Override
    public int compareTo(Component other) {
        Double thisRetTime = this.getRetTime();
        Double otherRetTime = other.getRetTime();
        return thisRetTime.compareTo(otherRetTime);
    }
    
    @Override
    public int compare(Component c1, Component c2) {
        Double rt1 = c1.getRetTime();
        Double rt2 = c2.getRetTime();    
        return Double.compare(rt1, rt2);
    }
    
    // ------------------------------------------------------------------------
    // ----- Properties -------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public List <Peak> getPeaks() {return peaks;}
    public NavigableMap <Double, Double> getScale() {return scale;}
    public NavigableMap <Double, Double> getSpectrum() {return spectrum;}
    public NavigableMap <Double, Double> getChromtogram()
        {return bestPeak.getChromatogram();}
    public Peak getBestPeak() {return bestPeak;}

    public int size() {return peaks.size();}
    public double getIntensity() {return bestPeak.getIntensity();}
    public double getRetTime() {return bestPeak.getRetTime();}
    public double getMZ() {return mz;}
    public void setMZ(double mz) {
        this.mz = mz;
    }
    
    public double getSpectrumNorm() {return norm;}
    public double getChromatogramNorm() {return bestPeak.getNorm();}
    
    public boolean getReferenceStatus() {return isReference;}
    public boolean getAlignedStatus() {return isAligned;}
    
    public double getShift() {return shift;}
    
    @Override
    public String toString() {return this.bestPeak.toString();}
}
