/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import dulab.adap.common.distances.Distance;
import dulab.adap.common.distances.PurityScore;
import dulab.adap.common.parsers.CompoundInfo;
import dulab.adap.common.parsers.Parser;
import dulab.adap.common.types.BallTree;
import dulab.adap.common.types.BallTreeSQLiteInterface;
import dulab.adap.common.types.SparseMatrix;
import dulab.adap.common.types.SparseVector;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

/**
 *
 * @author aleksandrsmirnov
 */
public class DataBase 
{   
    static private final int DEFAULT_DATA_FILE_SIZE = 1024;
    
    private final List <CompoundInfo> info = new ArrayList <> ();
    private final List <Map <Double, Double>> data = new ArrayList <> ();
    
    private final SparseMatrix matrix = new SparseMatrix();
    private BallTree tree = null;
    
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    
    
    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public void readDataFiles(String fileType, String ... fileNames) {
        readDataFiles(fileType, DEFAULT_DATA_FILE_SIZE, fileNames);
    }
    
    public void readDataFiles(
            String fileType, int expectedSize, String ... fileNames)
    {
        for (String fileName : fileNames) 
        {
            try {
                Parser parser = Parser.loadInstance(fileType, fileName);

                this.info.addAll(parser.info());
                this.data.addAll(parser.data());

            } catch (IOException e) {
                throw new IllegalArgumentException(
                        "Couldn't read file " + fileName + ": "
                        + e.getMessage());
            }
        }
    }
    
    
    public void processData(int distanceIdentifier) throws Exception
    {
        final int infoSize = this.info.size();
        final int dataSize = this.data.size();
        
        if (infoSize == 0 && dataSize == 0)
            throw new Exception("DataBase contains no data");
        
        if (infoSize != dataSize)
            throw new Exception("DataBase contains corrupted data");
        
        // ------------------------------
        // Transform data to SparseMatrix
        // ------------------------------
        
        for (int i = 0; i < dataSize; ++i)
        {
            SparseVector vector = new SparseVector(data.get(i));
            
            vector.setInfo(info.get(i));
            vector.multiply(1.0 / vector.norm(), false);
            
            this.matrix.add(vector);
        }
        
        // ----------------------
        // Choose a distance type
        // ----------------------
        
        Distance distance = Distance.loadInstance(distanceIdentifier);
        
        // --------------
        // Build BallTree
        // --------------
        
        this.tree = new BallTree(this.matrix, distance);
        this.tree.build();
    }
    
    public void saveTree(String fileName) throws Exception
    {
        BallTreeSQLiteInterface.saveBallTree(fileName, this.tree);
        /*FileOutputStream fos = null;
        ObjectOutputStream oos = null;
        try {
            fos = new FileOutputStream(fileName);
            oos = new ObjectOutputStream(fos);
            oos.writeObject(this.tree);
        } catch (IOException e) {
            throw new Exception("Couldn't save DataBase: " + e.getMessage());
        } finally {
            oos.close();
            fos.close();
        }*/
    }
    
    public void loadTree(String fileName) throws Exception {
        loadTree(fileName, false);
    }
    
    public void loadTree(String fileName, boolean force) throws Exception
    {
        if (this.tree != null && !force)
            throw new Exception("DataBase already contains a tree");
        
        this.tree = BallTreeSQLiteInterface.loadBallTree(fileName);
    }
    
    public List <SparseVector> search(
            final Map <Double, Double> spectrum, float tolerance)
    {
        SparseVector vector = new SparseVector(spectrum);
        vector.multiply(1f / vector.norm(), false);
        return this.tree.getCloseVectors(vector, tolerance);
    }
    
    public void search(final Map <Double, Double> spectrum, 
            float tolerance, List <SparseVector> result)
    {
        final SparseVector vector = new SparseVector(spectrum);
        vector.multiply(1f / vector.norm(), false);
        this.tree.getCloseVectors(vector, tolerance, result);
        
        final Distance distance = new PurityScore();
        
        Collections.sort(result, new Comparator <SparseVector> () {
            @Override
            public int compare(SparseVector v1, SparseVector v2) 
            {  
                double d1 = distance.call(vector, v1);
                double d2 = distance.call(vector, v2);
                
                return Double.compare(d1, d2);
            }
        });
    }
    
    // ------------------------------------------------------------------------
    // ----- Properties -------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public int size() {return this.data.size();}
    
    public double getProgress() {
        if (this.tree != null)
            return this.tree.getProgress();
        
        return 0.0;
    }
    
    public BallTree tree() {return this.tree;}
}
