/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import dulab.adap.common.algorithms.Combinatorics;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.math3.util.CombinatoricsUtils;

/**
 *
 * @author aleksandrsmirnov
 */
public class IsotopicDistribution 
{   
    private Map <Double, Double> data;
 
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    public IsotopicDistribution() {
        data = new HashMap <> ();
    }

    public IsotopicDistribution(Map <Double, Double> data) {
        this.data = data;
    }

    public IsotopicDistribution(IsotopicDistribution id) {
        data = new HashMap <> (id.data);
    }

    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    private double round(double number) {
        return Math.round(1000000 * number) / 1000000.0;
    }
    
    public void put(double key, double value) {
        data.put(key, value);
    }

    public double get(double key) {
        Double value = data.get(key);
        if (value == null) value = 0.0;
        return value;
    }

    public Map <Double, Double> data() {
        return data;
    }
    
    public void scale(double upper_boundary) 
    {
        double maxIntensity = 0.0;

        for (double intensity : this.data.values())
            if (intensity > maxIntensity)
                maxIntensity = intensity;

        double scale = upper_boundary / maxIntensity;

        for (Map.Entry <Double, Double> e : this.data.entrySet())
            e.setValue(scale * e.getValue());
    }
    
    public IsotopicDistribution multiply(IsotopicDistribution other) 
    {   
        if (this.data.isEmpty()) return new IsotopicDistribution(other);
        if (other.data.isEmpty()) return new IsotopicDistribution(this);

        IsotopicDistribution result = new IsotopicDistribution();

        for (Map.Entry <Double, Double> isotope1 : this.data.entrySet())
            for (Map.Entry <Double, Double> isotope2 : other.data.entrySet())
            {
                double mass = round(isotope1.getKey() + isotope2.getKey());
                result.put(mass, result.get(mass) 
                        + isotope1.getValue() * isotope2.getValue());
            }

        return result;
    }

    public IsotopicDistribution power(int n)
    {
        this.scale(1.0);

        int k = this.data.size();
        List <int[]> combinations = 
                Combinatorics.getCombinations(n, this.data.size());

        IsotopicDistribution result = new IsotopicDistribution();
        //double lnfactorial_n = CombinatoricsUtils.factorialLog(n + 1);
        double lnfactorial_n = CombinatoricsUtils.factorialLog(n);

        for (int[] tuple : combinations)
        {
            double lnintensity = lnfactorial_n;
            double mass = 0.0;

            Iterator <Map.Entry <Double, Double>> it = 
                    this.data.entrySet().iterator();

            for (int i = 0; i < tuple.length; ++i)
            {
                Map.Entry <Double, Double> isotope = it.next();

                lnintensity += tuple[i] * Math.log(isotope.getValue())
                        //- CombinatoricsUtils.factorialLog(tuple[i] + 1);
                        - CombinatoricsUtils.factorialLog(tuple[i]);

                mass += tuple[i] * isotope.getKey();
            }

            mass = round(mass);
            result.put(mass, result.get(mass) + Math.exp(lnintensity));
        }

        return result;
    }
}

