/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

/**
 *
 * @author aleksandrsmirnov
 */
public class IsotopicDistributionParser 
{   
    // ------------------------------------------------------------------------
    // ----- ISOTOPE_TABLE Initialization -------------------------------------
    // ------------------------------------------------------------------------
    
    private static final double NULL = 1e-300;
    private static final double PROTONE = 1.0078246;
    
    private static Map <Double, Double> map(Double ... values) {
        Map <Double, Double> result = new HashMap <> ();
        
        int size = values.length / 2;
        for (int i = 0; i < size; ++i)
            result.put(values[2 * i], values[2 * i + 1]);
        
        return Collections.unmodifiableMap(result);
    }
    
    private static Map <String, Map <Double, Double>> isotopeInitializer()
    {
        Map <String, Map <Double, Double>> result = new HashMap <> ();
        
        result.put("H", map(1.0078246, 99.9855, 2.0078246, 0.0145));
        result.put("C", map(12.0, 98.94, 13.0, 1.06));
        result.put("N", map(14.003074, 99.6205, 15.003074, 0.3795));
        result.put("O", map(15.994915, 99.757, 16.994915, 0.03835, 17.994915, 0.2045));
        result.put("S", map(31.972072, 94.85, 32.972072, 0.763, 33.972072, 4.365,
                34.972072, NULL, 35.972072, 0.0158));
        result.put("Cl", map(34.968853, 75.8, 35.968853, NULL, 36.968853, 24.2));
        result.put("Br", map(78.918336, 50.65, 79.918336, NULL, 80.918336, 49.35));
        result.put("Si", map(27.976928, 92.2545, 28.976928, 4.672, 29.976928, 3.0735));
        result.put("P", map(30.97376, 100.0));
        result.put("F", map(18.9984, 100.0));
        result.put("I", map(126.90448, 100.0));
        result.put("Mg", map(23.9850535, 78.965, 24.9850535, 10.011, 25.9850535, 11.025));
        result.put("Fe", map(53.9441635, 5.845105, 54.9441635, NULL, 55.9441635, 
                91.754106, 56.9441635, 2.11929, 57.9441635, 0.28212));
        result.put("Mo", map(91.9113635, 14.649106, 92.9113635, NULL, 93.9113635, 
                9.18733, 94.9113635, 15.8733, 95.9113635, 16.6738, 96.9113635, 
                9.58215, 97.9113635, 24.2928, 98.9113635, NULL, 99.9113635, 9.74465));
        result.put("Mn", map(54.9426035, 100.0));
        result.put("Cu", map(62.9441535, 69.1515, 63.9441535, NULL, 64.9441535, 30.8515));
        result.put("B", map(10.0174535, 19.65, 11.0174535, 80.35));
        result.put("Ca", map(39.9671435, 96.941156, 40.9671435, NULL, 41.9671435, 
                0.64723, 42.9671435, 0.1351, 43.9671435, 2.08611, 44.9671435, 
                NULL, 45.9671435, 0.0043, 46.9671435, NULL, 47.9671435, 0.18721));
        result.put("Ni", map(57.9399035, 68.076919, 58.9399035, NULL, 59.9399035, 
                26.223115, 60.9399035, 1.139913, 61.9399035, 3.63454, 62.9399035, 
                NULL, 63.9399035, 0.925619));
        result.put("Zn", map(63.9336935, 49.1775, 64.9336935, NULL, 65.9336935, 
                27.7398, 66.9336935, 4.0416, 67.9336935, 18.4563, 68.9336935, NULL, 
                69.9336935, 0.611));
        result.put("Co", map(58.9377535, 100.0));
        
        return Collections.unmodifiableMap(result);
    }
    
    private static final Map <String, Map <Double, Double>> ISOTOPE_TABLE =
            isotopeInitializer();
    
    // ------------------------------------------------------------------------
    // ----- Molecular Formula Parser -----------------------------------------
    // ------------------------------------------------------------------------
    
    private static final int ascii_a = (int) 'a';
    private static final int ascii_A = (int) 'A';
    private static final int ascii_0 = (int) '0';
    
    private static Map <String, Integer> getFormula(String formula)
    {
        Map <String, Integer> result = new HashMap <> ();
        
        String atom = "", quantity = "";
        
        for (final char character : formula.toCharArray())
        {
            int ascii = (int) character;
            
            if (ascii >= ascii_a) // Part of the atom name
                atom += character;
            else if (ascii >= ascii_A) // New atom
            { 
                if (atom.length() > 0)  // Save previous atom
                {
                    result.put(atom,
                        quantity.length() > 0 ? Integer.parseInt(quantity) : 1);
                    
                    atom = ""; 
                    quantity = "";
                }
                atom += character; // Start new atom
            }
            else if (ascii >= ascii_0)
                quantity += character;
        }
        
        // Add the last atom
        if (atom.length() > 0)
            result.put(atom,
                    quantity.length() > 0 ? Integer.parseInt(quantity) : 1);
        
        return result;
    }
    
    /**
     * Builds IsotopicDistribution for the given molecular formula
     * @param formula molecular formula
     * @return IsotopicDistribution
     */
    
    public static IsotopicDistribution call(String formula, boolean addProtone)
    {
        Map <String, Integer> mapFormula = getFormula(formula);
        
        IsotopicDistribution result = new IsotopicDistribution();
        
        for (Entry <String, Integer> entry : mapFormula.entrySet())
        {
            String atom = entry.getKey();
            int quantity = entry.getValue();
            
            Map <Double, Double> iso = ISOTOPE_TABLE.get(atom);
            if (iso == null) return null;
            
            IsotopicDistribution distribution = 
                    new IsotopicDistribution(new HashMap <> (iso));
            
            distribution.scale(1.0);
            
            result = result.multiply(distribution.power(quantity));
        }
        
        result.scale(1.0);
        
        if (addProtone) {
            Map <Double, Double> data = new HashMap <> ();
            for (Entry <Double, Double> e : result.data().entrySet())
                data.put(e.getKey() + PROTONE, e.getValue());
            
            result = new IsotopicDistribution(data);
        }
        
        return result;
    }
}
