/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import dulab.adap.common.distances.Distance;
import dulab.adap.common.distances.DotProductDistance;
import dulab.adap.common.parsers.CompoundInfo;
import dulab.adap.common.types.SparseVector;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 *
 * @author aleksandrsmirnov
 */
public class LCMSDataBase {
    
    Connection connection;
    
    // ------------------------------------------------------------------------
    // ----- Consructors ------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public LCMSDataBase(final String filename)
    {
        try {
            connection = DriverManager.getConnection("jdbc:sqlite:" + filename);
        } catch (SQLException e) {
            throw new IllegalArgumentException("Couldn't open SQLite database " 
                    + filename + ": " + e.getMessage());
        }    
    }
    
    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public List <SparseVector> search(
            final double mass, final Map <Double, Double> spectrum, 
            final double massTolerance, final double distanceTolerance)
    {   
        class VectorDistancePair {
            SparseVector vector;
            double distance;
        }
        
        List <VectorDistancePair> bestPairs = new ArrayList <> ();
        
        Distance distance = new DotProductDistance();
        
        double minMass = mass - massTolerance;
        double maxMass = mass + massTolerance;
        
        SparseVector predictedVector = new SparseVector(new TreeMap(spectrum));
        
        // -----------------------
        // Read data from DataBase
        // -----------------------
        
        try {
            Statement s = connection.createStatement();
            
            ResultSet rs = s.executeQuery(
                    "SELECT * FROM compounds "
                    + "WHERE mass > " + minMass + " AND mass < " + maxMass);
            
            while (rs.next()) 
            {    
                String formula = rs.getString("formula");
                
                if (formula.contains("+") || formula.contains("-"))
                    continue;
                
                //System.out.println(formula);
                
                IsotopicDistribution distribution =
                        IsotopicDistributionParser.call(formula, true);
                
                if (distribution == null) {
                    System.out.println("Skipping formula " + formula + " ...");
                    continue;
                }
                
                SparseVector v = new SparseVector(new TreeMap(distribution.data()));
                v.trim(0.001); // Remove intensities below 0.001
                
                // Calcuate distance and compare to others
                
                double d = distance.call(v, predictedVector, massTolerance);
                
                if (d < distanceTolerance) {
                    v.setInfo(new CompoundInfo()
                        .id("PubChem ID: " + Long.toString(rs.getLong("id")))
                        .formula(formula)
                        .name(rs.getString("name")));
                    
                    VectorDistancePair pair = new VectorDistancePair();
                    pair.vector = v;
                    pair.distance = d;
                    bestPairs.add(pair);
                }
            }
            
            Collections.sort(bestPairs, new Comparator <VectorDistancePair> () {
                @Override
                public int compare(VectorDistancePair p1, VectorDistancePair p2) {
                    return Double.compare(p1.distance, p2.distance); // Sort pairs in ascending order
                }
            });
            
        } catch (SQLException e) {
            throw new IllegalArgumentException("Unsuccesfull database query " +
                    e.getMessage());
        }
        
        // Return list of found components
        
        List <SparseVector> result = new ArrayList <> (bestPairs.size());
        
        for (VectorDistancePair pair : bestPairs)
            result.add(pair.vector);
        
        //while (!bestPairs.isEmpty())
        //    result.add(0, bestPairs.poll().vector);
        
        return result;
    }
    
}
