/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import com.google.common.collect.Range;
import dulab.adap.common.algorithms.Math;

import javax.annotation.Nonnull;
import java.io.Serializable;

import java.util.NavigableMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * Class Peak contains all information about a peak as well as
 * its chromatogram, and some methods to asList the information
 * 
 * @author aleksandrsmirnov
 */
public class Peak implements Cloneable, Serializable
{
    /** Retention time of the peak apex */
    private final double apexRetTime;

    /** Retention time range of the peak */
    private final double startRetTime;
    private final double endRetTime;

    /** M/z value of the peak apex */
    private final double apexMZ;

    /** M/z range of the peak */
    private final Range<Double> mzRange;

    /** Intensity of the peak apex */
    private final double apexIntensity;

    /** Chromatogram of the peak */
    private final NavigableMap <Double, Double> chromatogram; // (retTime, intensity) - pairs

    /** Norm of the chromatogram */
    private final double norm;

    /** PeakInfo object containing extra information */
    private PeakInfo info;

    /** Shift of the aligned peak */
    private double shift;
    
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    public Peak(final Peak peak) {
        info = new PeakInfo(peak.info);
        shift = peak.shift;
        chromatogram = new TreeMap <> (peak.chromatogram);

        startRetTime = peak.startRetTime;
        endRetTime = peak.endRetTime;
        mzRange = Range.closed(peak.mzRange.lowerEndpoint(), peak.mzRange.upperEndpoint());
        
        apexIntensity = peak.apexIntensity;
        apexRetTime = peak.apexRetTime;
        apexMZ = peak.apexMZ;
        
        norm = peak.norm;
    }
    
    public Peak(final NavigableMap <Double, Double> chromatogram, 
            final PeakInfo info) 
    {
        this(chromatogram, info.mzValue);
        
        //this.info = new PeakInfo(info);
        this.info = info;
    }
    
    public Peak(final NavigableMap <Double, Double> chromatogram, 
            final double mz) 
    {   
        this.info = new PeakInfo();
        this.info.mzValue = mz;

        this.shift = 0.0;
        this.chromatogram = new TreeMap <> (chromatogram);

        double apexRetTime = 0.0;
        double retTimeMin = Double.MAX_VALUE;
        double retTimeMax = 0.0;

        double apexIntensity = 0.0;

        double norm2 = 0.0;
        double prevRetTime = 0.0;
        double prevIntensity = 0.0;

        for (Map.Entry<Double, Double> entry : chromatogram.entrySet())
        {
            double retTime = entry.getKey();
            double intensity = entry.getValue();

            if (entry != chromatogram.firstEntry()) {
                double delta = retTime - prevRetTime;
                double height = 0.5 * (prevIntensity * prevIntensity + intensity * intensity);
                norm2 += delta * height;
            }
            
            if (retTime > retTimeMax) retTimeMax = retTime;
            if (retTime < retTimeMin) retTimeMin = retTime;

            if (intensity > apexIntensity) {
                apexIntensity = intensity;
                apexRetTime = retTime;
            }

            prevRetTime = retTime;
            prevIntensity = intensity;
        }

        this.apexRetTime = apexRetTime;
        this.startRetTime = retTimeMin;
        this.endRetTime = retTimeMax;

        this.apexMZ = mz;
        this.mzRange = Range.singleton(mz);

        this.apexIntensity = apexIntensity;

        this.norm = java.lang.Math.sqrt(Math
                .continuous_dot_product(chromatogram, chromatogram));
//        this.norm = java.lang.Math.sqrt(norm2);
    }
    
    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public void setShift(double shift) {this.shift = shift;};
    
    @Override
    public Peak clone() {return new Peak(chromatogram, info);}

    @Nonnull
    public Peak merge(Peak other) {
        NavigableMap<Double, Double> chromatogram = new TreeMap<>(this.chromatogram);
        chromatogram.putAll(other.chromatogram);
        return new Peak(chromatogram, PeakInfo.merge(this.info, other.info));
    }
    
    // ------------------------------------------------------------------------
    // ----- Properties -------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public NavigableMap <Double, Double> getChromatogram() {return chromatogram;};
    public PeakInfo getInfo() {return info;};
    
    public double getRetTime() {return apexRetTime;}
    public double getStartRetTime() {return startRetTime;}
    public double getEndRetTime() {return endRetTime;}
    public double getMZ() {return apexMZ;}
    public Range<Double> getMzRange() {return mzRange;}
    public double getIntensity() {return apexIntensity;}
    
    public double getNorm() {return norm;}
    
    @Override
    public String toString() {
        return "#" + this.info.peakID + ": mz=" + this.apexMZ + " rt=" + this.apexRetTime;
    }
}
