/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import java.security.InvalidParameterException;
import java.util.NavigableMap;
import java.util.TreeMap;

/**
 *
 * @author aleksandrsmirnov
 */
public class RawData {
    public double[] retTimes;
    public double[] intensities;
    public double[] mzValues;
    
    /**
     * Create a DataPointCollection from parameters such that
     * 
     * size(intensities) = size(retTimes) * size(mzValues)
     * 
     * @param retTimes list of retention times
     * 
     * @param intensities list of intensities
     * 
     * @param mzValues list of m/z values 
     */
    
    public RawData(final double[] retTimes, final double[] intensities,
            final double[] mzValues) 
    {
        if (intensities.length == retTimes.length * mzValues.length) {
            this.retTimes = retTimes;
            this.intensities = intensities;
            this.mzValues = mzValues;
        } else {
            this.retTimes = new double[0];
            this.intensities = new double[0];
            this.mzValues = new double[0];
        
            throw new InvalidParameterException(
                    "Parameters should satisfy the condition "
                            + "size(intensities) = size(retTimes) * size(mzValues)");
        }
    }
    
    public NavigableMap <Double, Double> getChromatogram(
            final int start, final int finish, final int offset)
    {
        NavigableMap <Double, Double> result = new TreeMap <> ();
        
        for (int i = start; i <= finish; ++i)
            result.put(retTimes[i], intensities[i + offset]);
        
        return result;
    }
}
