/* 
 * Copyright (C) 2016 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.datamodel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map.Entry;
import java.util.NavigableMap;
import java.util.stream.IntStream;

/**
 *
 * @author aleksandrsmirnov
 */
public class ReferenceComponent extends Component {
    
    private int sampleID;
    private final double score;
    private final List <Component> alignedComponents = new ArrayList <> ();
    private final List <Integer> sampleIDs = new ArrayList <> ();
    
    // ------------------------------------------------------------------------
    // ----- Constructors -----------------------------------------------------
    // ------------------------------------------------------------------------
    
    public ReferenceComponent(Component component, double score) {
        super(component);
        this.score = score;
    }
    
    // ------------------------------------------------------------------------
    // ----- Methods ----------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public void addComponent(final Component component, final int id) {
        alignedComponents.add(new Component(component));
        sampleIDs.add(id);
    }
    
    public void setSampleID(final int id) {this.sampleID = id;}

    // ------------------------------------------------------------------------
    // ----- Properties -------------------------------------------------------
    // ------------------------------------------------------------------------
    
    public List <Integer> getSampleIDs() {return this.sampleIDs;}
    public int getSampleID() {return this.sampleID;};
    public int getSampleID(final int n) {return this.sampleIDs.get(n);}
    
    public List <Component> getComponents() {return this.alignedComponents;}
    public Component getComponent(final int n) {
        return this.alignedComponents.get(n);
    }
    
    public double getScore() {return this.score;}
    
    @Override
    public int size() {return this.alignedComponents.size();}

    /**
     * Finds the quantitation mass and assign it to all components.
     *
     * Starting with the highest component and the highest peak in its spectrum, every m/z is
     * checked for being present in all other spectra. If so, then that m/z is used as quantitation
     * mass. Otherwise, the next peak is checked in the intensity-descending order.
     *
     * @param mzTolerance tolerance when two m/z values are considered matched
     */
    public void adjustMasses(double mzTolerance) {

        double[] mzValues = new double[alignedComponents.size()];

        // First try to select the quantitation m/z from the components' m/z values
        if (selectQuantitationMzFromComponentMz(mzTolerance, mzValues)) {
            saveQuantitationMzValues(mzValues);
        }

        // If unsuccessful, select the quantitation m/z from all m/z values in all spectra
        else if (selectQuantitationMzFromSpectrumMz(mzTolerance, mzValues)) {
            saveQuantitationMzValues(mzValues);
        }
    }

    private boolean selectQuantitationMzFromComponentMz(
        double tolerance, double[] quantitationMzValues) {

        // Collect all components' m/z values
        double[] mzValues = alignedComponents.stream()
            .mapToDouble(Component::getMZ)
            .distinct()
            .toArray();

        // Count the number of components for each m/z value
        long[] mzCounts = Arrays.stream(mzValues)
            .mapToLong(queryMz -> alignedComponents
                .stream()
                .mapToDouble(Component::getMZ)
                .filter(mz -> Math.abs(mz - queryMz) < tolerance)
                .count())
            .toArray();

        // Sort in the count-descending order
        int[] sortIndices = IntStream.range(0, mzCounts.length)
            .boxed()
            .sorted(Comparator.comparingLong(i -> -mzCounts[i]))
            .mapToInt(Integer::intValue)
            .toArray();

        // Check every m/z value on presence in all spectra
        for (int index : sortIndices) {
            double mz = mzValues[index];
            if (presentInAllComponents(mz, tolerance, quantitationMzValues))
                return true;
        }

        return false;
    }

    private boolean selectQuantitationMzFromSpectrumMz(
        double tolerance, double[] quantitationMzValues) {

        // Sort all components in the intensity-descending order
        int[] componentIndices = IntStream.range(0, alignedComponents.size())
            .boxed()
            .sorted(Comparator.comparingDouble(i -> -alignedComponents.get(i).getIntensity()))
            .mapToInt(Integer::intValue)
            .toArray();

        // Iterate over components starting with the highest
        for (int componentIndex : componentIndices) {

            Component component = alignedComponents.get(componentIndex);
            NavigableMap<Double, Double> spectrum = component.getSpectrum();

            // Convert a spectrum into arrays of m/z values and intensities.
            // Array of indices contains the indices of the first two arrays in the
            // intensity-descending order.
            double[] mzValues = new double[spectrum.size()];
            double[] intensities = new double[spectrum.size()];
            int[] indices = new int[spectrum.size()];
            spectrumToArrays(spectrum, mzValues, intensities, indices);

            // Iterate over spectrum peaks starting with the highest peak
            for (int index : indices) {

                double mz = mzValues[index];

                // If quantitationMz is present in all components, finish the procedure
                if (presentInAllComponents(mz, tolerance, quantitationMzValues)) {
                    return true;
                }
            }
        }

        return false;
    }

    private boolean presentInAllComponents(
        double quantitationMz, double tolerance, double[] mzValues) {

        // Check the presence of quantitationMz in all components
        for (int i = 0; i < alignedComponents.size(); ++i) {

            // Find the closest m/z within tolerance
            double closestMz = Double.NaN;
            double minDistance = Double.MAX_VALUE;
            for (double mz : alignedComponents.get(i).getSpectrum().keySet()) {
                double d = Math.abs(mz - quantitationMz);
                if (d < tolerance && d < minDistance) {
                    minDistance = d;
                    closestMz = mz;
                    mzValues[i] = mz;
                }
            }

            // If quantitationMz is missing in the current component, stop the check
            if (Double.isNaN(closestMz)) {
                return false;
            }
        }

        return true;
    }

    private void saveQuantitationMzValues(double[] mzValues) {

        for (int i = 0; i < alignedComponents.size(); ++i) {

            double mz = mzValues[i];
            Component component = alignedComponents.get(i);

            double newIntensity = component.getSpectrum().get(mz);
            double scale = newIntensity / component.getIntensity();

            for (Entry<Double, Double> entry : component.getChromtogram().entrySet())
                entry.setValue(scale * entry.getValue());

            component.setMZ(mz);
        }
    }

    private void spectrumToArrays(NavigableMap<Double, Double> spectrum,
        double[] mzValues, double[] intensities, int[] indices) {

        int index = 0;
        for (Entry<Double, Double> e : spectrum.entrySet()) {
            mzValues[index] = e.getKey();
            intensities[index] = e.getValue();
            ++index;
        }

        int[] tmpIndices = IntStream.range(0, spectrum.size())
            .boxed()
            .sorted(Comparator.comparingDouble(i -> -intensities[i]))
            .mapToInt(Integer::intValue)
            .toArray();

        System.arraycopy(tmpIndices, 0, indices, 0, tmpIndices.length);
    }
}
