/*
 * Copyright (C) 2018 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.workflow.decomposition;

import org.jblas.DoubleMatrix;
import org.jblas.MatrixFunctions;

import javax.annotation.Nonnull;

/**
 * @author Du-Lab Team dulab.binf@gmail.com
 */
public class BiGaussian
{
    private static final double EPS = 1e-12;

    final double height;
    final double center;
    final double leftSTD;
    final double rightSTD;

    BiGaussian(double h, double c, double ls, double rs) {
        height = h;
        center = c;
        leftSTD = ls;
        rightSTD = rs;
    }

    static double evaluate(double x, @Nonnull BiGaussian biGaussian) {
        return evaluate(x, biGaussian.height, biGaussian.center, biGaussian.leftSTD, biGaussian.rightSTD);
    }

    static double evaluate(double x, double h, double c, double ls, double rs) {
        double d = x - c;
        double s = d > 0.0 ? rs : ls;
        return h * Math.exp(-d * d / (2 * s * s));
    }

    static DoubleMatrix evaluate(@Nonnull DoubleMatrix x, double h, double c, double ls, double rs) {
        DoubleMatrix d = x.sub(c);
        DoubleMatrix s = new DoubleMatrix(x.rows, x.columns);
        s.put(d.ge(0.0), rs);
        s.put(d.le(0.0), ls);
        return MatrixFunctions.exp(d.mul(d).mul(-1).div(s.mul(s).mul(2).max(EPS))).mul(h);
    }

    static DoubleMatrix evaluate(@Nonnull DoubleMatrix x, @Nonnull BiGaussian biGaussian) {
        return evaluate(x, biGaussian.height, biGaussian.center, biGaussian.leftSTD, biGaussian.rightSTD);
    }

    static double[] evaluate(@Nonnull double[] x, @Nonnull BiGaussian biGaussian) {
        return evaluate(new DoubleMatrix(x), biGaussian).toArray();
    }
}
