/*
 * Copyright (C) 2017 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dulab.adap.workflow.decomposition;

import dulab.adap.datamodel.BetterComponent;
import dulab.adap.datamodel.BetterPeak;

import javax.annotation.Nonnull;
import java.util.*;

/**
 * Detects the presence of and constructs components, then performs decomposition of chromatograms into
 * the linear combination of those components.
 *
 * @author Du-Lab Team dulab.binf@gmail.com
 */
public class Decomposition {

    // Used to calculate the algorithm progress
    private double processedPercent = 0.0;

    // Used to stop the algorithm
    private boolean isCancelled = false;

    /**
     * Returns the current progress of the algorithm.
     * @return the progress
     */
    public double getProcessedPercent() {
        return processedPercent;
    }

    /**
     * Stops the algorithms from running.
     */
    public void cancel() {isCancelled = true;}
    
    /**
     * Spectral deconvolution of chromatograms.
     *
     * First EIC peak detection is performed. Then the presence of components is detected and their elution profiles
     * are formed by the non-negative matrix factorization. Finally, all chromatograms are decomposed into the
     * linear combination of the components and the baseline.
     * 
     * @param params instance of {@link Parameters} with decomposition parameters
     * @param chromatograms list of chromatograms
     * @return a list of {@link BetterComponent} with constructed components and their spectra
     */
    @Nonnull
    public List<BetterComponent> run(@Nonnull final Parameters params,
                                     @Nonnull final List<BetterPeak> chromatograms,
                                     @Nonnull final List<BetterPeak> peaks)
    {
        final double processedStep = 1.0;
        processedPercent = 0.0;

        // -------------------------------------------------------------
        // Cluster peaks based on their apex and boundary retention times
        // -------------------------------------------------------------

        RetTimeClusterer clusterer = new RetTimeClusterer(params.prefWindowWidth);
        List<RetTimeClusterer.Cluster> retTimeClusters = clusterer.execute(peaks);

        // -------------------------------------
        // Within each cluster, find components
        // -------------------------------------

        final double localProcessedStep = processedStep / retTimeClusters.size();

        ComponentSelector componentSelector = new ComponentSelector();
        List<BetterComponent> components = new ArrayList<>();
        for (RetTimeClusterer.Cluster cluster : retTimeClusters)
        {
            if (isCancelled) {
                components.clear();
                break;
            }

            components.addAll(componentSelector.execute(chromatograms, cluster,
                    params.retTimeTolerance, params.adjustApexRetTimes, params.minClusterSize));
            
            processedPercent += localProcessedStep;
        }

        return components;
    }

    /**
     * Stores parameters of the decomposition
     */
    public static class Parameters
    {
        /** Preferable width of the deconvolution windows */
        public double prefWindowWidth = 0.5;

        /** Retention-time tolerance between components */
        public double retTimeTolerance = 0.05;

        public int minClusterSize = 1;

        public boolean adjustApexRetTimes = false;
    }
}
