package dulab.adap.workflow.decomposition;

/**
 * This is a helper class to work with Matrices
 */
public class MyMatrix {

    private final double[][] data;
    private final int numRows;
    private final int numCols;

    /**
     * Creates an instance of MyMatrix with the given data
     *
     * @param data 2D-array
     */
    public MyMatrix(double[][] data) {
        this.data = data;

        this.numRows = data.length;
        int numCols = 0;
        for (int i = 0; i < numRows; ++i)
            if (data[i].length > numCols)
                numCols = data[i].length;
        this.numCols = numCols;
    }

    /**
     * Creates an instance of MyMatrix with zero values
     *
     * @param numRows number of rows
     * @param numCols number of columns
     */
    public MyMatrix(int numRows, int numCols) {
        this(new double[numRows][numCols]);
    }

    public double[][] getData() {
        return data;
    }

    public int getNumRows() {
        return numRows;
    }

    public int getNumCols() {
        return numCols;
    }

    public double get(int i, int j) {
        return data[i][j];
    }

    public double[] getColumn(int colIndex) {
        double[] column = new double[numRows];
        for (int i = 0; i < numRows; ++i)
            column[i] = data[i][colIndex];
        return column;
    }

    public double[] getRow(int rowIndex) {
        return data[rowIndex];
    }

    public void set(int i, int j, double x) {
        data[i][j] = x;
    }

    public void set(MyMatrix m) {
        int minNumRows = Math.min(numRows, m.numRows);
        int minNumCols = Math.min(numCols, m.numCols);
        for (int i = 0; i < minNumRows; ++i)
            for (int j = 0; j < minNumCols; ++j)
                data[i][j] = m.get(i, j);
    }

    /**
     * Creates a transposed matrix
     *
     * @return a transposed matrix
     */
    public MyMatrix transpose() {
        double[][] array = new double[numCols][numRows];
        for (int i = 0; i < numRows; ++i)
            for (int j = 0; j < numCols; ++j)
                array[j][i] = data[i][j];
        return new MyMatrix(array);
    }

    /**
     * Divides each column by the square root of its maximum
     */
    public void divideColumnsBySquareRootMax() {
        for (int j = 0; j < numCols; ++j) {
            double max = findColumnMaximum(j);
            double factor = Math.sqrt(max);
            for (int i = 0; i < numRows; ++i)
                data[i][j] /= factor;
        }
    }

    /**
     * Divides each column by its maximum
     */
    public void divideColumnsByMax() {
        for (int j = 0; j < numCols; ++j) {
            double max = findColumnMaximum(j);
            for (int i = 0; i < numRows; ++i)
                data[i][j] /= max;
        }
    }

    /**
     * Creates a new matrix with an extra column of values x
     *
     * @param x value
     * @return a new matrix
     */
    public MyMatrix appendColumn(double x) {
        double[][] array = new double[numRows][numCols + 1];
        for (int i = 0; i < numRows; ++i) {
            System.arraycopy(data[i], 0, array[i], 0, numCols);
            array[i][numCols] = x;
        }
        return new MyMatrix(array);
    }

    /**
     * Creates a new matrix with an extra row of values x
     *
     * @param x value
     * @return a new matrix
     */
    public MyMatrix appendRow(double x) {
        double[][] array = new double[numRows + 1][numCols];
        for (int i = 0; i < numRows; ++i)
            System.arraycopy(data[i], 0, array[i], 0, numCols);
        for (int j = 0; j < numCols; ++j)
            array[numRows][j] = x;
        return new MyMatrix(array);
    }

    /**
     * Find the maximum of values in a row
     *
     * @param rowIndex index of a row
     * @return the maximum
     */
    public double findRowMaximum(int rowIndex) {
        return data[rowIndex][findIndexOfRowMaximum(rowIndex)];
    }

    /**
     * Finds index of the maximum of values in a row
     *
     * @param rowIndex index of a row
     * @return index of the maximum
     */
    public int findIndexOfRowMaximum(int rowIndex) {
        double max = -Double.MAX_VALUE;
        int colIndex = 0;
        for (int j = 0; j < numCols; ++j)
            if (data[rowIndex][j] > max) {
                max = data[rowIndex][j];
                colIndex = j;
            }
        return colIndex;
    }

    /**
     * Find the maximum of values in a column
     *
     * @param colIndex index of a column
     * @return the maximum
     */
    public double findColumnMaximum(int colIndex) {
        return data[findIndexOfColumnMaximum(colIndex)][colIndex];
    }

    /**
     * Find index of the maximum of values in a column
     *
     * @param colIndex index of a column
     * @return index of the maximum
     */
    public int findIndexOfColumnMaximum(int colIndex) {
        double max = -Double.MAX_VALUE;
        int rowIndex = 0;
        for (int i = 0; i < numRows; ++i)
            if (data[i][colIndex] > max) {
                max = data[i][colIndex];
                rowIndex = i;
            }
        return rowIndex;
    }

    /**
     * Finds the maximum of all elements
     *
     * @return the maximum
     */
    public double findMaximum() {
        double max = -Double.MAX_VALUE;
        for (int i = 0; i < numRows; ++i)
            for (int j = 0; j < numCols; ++j)
                if (data[i][j] > max)
                    max = data[i][j];
        return max;
    }

    /**
     * Creates a new matrix without zero columns
     *
     * @return a new matrix
     */
    public MyMatrix removeZeroColumns() {

        boolean[] keep = new boolean[numCols];
        int numNonZeroCols = 0;
        for (int j = 0; j < numCols; ++j) {
            if (findColumnMaximum(j) > 0.0) {
                keep[j] = true;
                ++numNonZeroCols;
            }
        }

        double[][] array = new double[numRows][numNonZeroCols];
        for (int i = 0; i < numRows; ++i) {
            int colIndex = 0;
            for (int j = 0; j < numCols; ++j)
                if (keep[j]) {
                    array[i][colIndex] = data[i][j];
                    ++colIndex;
                }
        }

        return new MyMatrix(array);
    }
}
