/*
 * Copyright (C) 2018 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.workflow.decomposition;

import dulab.adap.datamodel.Chromatogram;
import org.jblas.DoubleMatrix;
import org.jblas.Solve;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.Comparator;
import java.util.stream.IntStream;

/**
 * Stores information about a peak including chromatogram (can be longer than the peak), m/z value of the peak,
 * indices of the peak boundaries, index of the peak apex, and the apex retention time calculated by fitting a
 * parabola
 */
public class Peak {

    final Chromatogram chromatogram;
    final int apexIndex;
    final int startIndex;
    final int endIndex;
    final double halfMaxStartRetTime;
    final double halfMaxEndRetTime;
    final double apexRetTime;
    final double mzValue;

    Peak(@Nonnull Chromatogram c, double startRetTime, double endRetTime, double mz, boolean adjustApexRetTime)
            throws IllegalArgumentException {

        this.mzValue = mz;
        this.chromatogram = c;
        this.startIndex = search(c.xs, startRetTime);
        this.endIndex = search(c.xs, endRetTime);
        this.apexIndex = IntStream.rangeClosed(startIndex, endIndex).boxed()
                .max(Comparator.comparing(i -> c.ys[i]))
                .orElse((startIndex + endIndex) / 2);

        // Find indices of the top part
        double halfHeight = 0.5 * c.ys[apexIndex];
        int ind1 = apexIndex;
        while (ind1 > startIndex && c.ys[ind1 - 1] > halfHeight) --ind1;
        int ind2 = apexIndex;
        while (ind2 < endIndex && c.ys[ind2 + 1] > halfHeight) ++ind2;

        this.halfMaxStartRetTime = (ind1 <= startIndex) ? c.xs[ind1] : getX(c, ind1 - 1, ind1, halfHeight);
        this.halfMaxEndRetTime = (ind2 >= endIndex) ? c.xs[ind2] : getX(c, ind2, ind2 + 1, halfHeight);

        double retTime = c.xs[apexIndex];

        if (adjustApexRetTime) {
            // Construct matrices for fitting parabola
            final int numRows = ind2 - ind1 + 1;
            DoubleMatrix a = DoubleMatrix.ones(numRows, 3);
            for (int i = 0; i < numRows; ++i) {
                a.put(i, 0, c.xs[ind1 + i] * c.xs[ind1 + i]);
                a.put(i, 1, c.xs[ind1 + i]);
            }
            DoubleMatrix b = new DoubleMatrix(numRows, 1,
                    Arrays.copyOfRange(c.ys, ind1, ind2 + 1));

            // Fit parabola and find the apex retention time

            try {
                DoubleMatrix params = Solve.solveLeastSquares(a, b);
                retTime = -params.get(1) / (2 * params.get(0));
                if (params.get(0) >= 0 || retTime < c.xs[ind1] || retTime > c.xs[ind2])
                    retTime = c.xs[apexIndex];

            } catch (Exception e) {
                retTime = c.xs[apexIndex];
            }
        }

        this.apexRetTime = retTime;
    }

    /**
     * Finds index of the element equal to key.
     *
     * @param array sorted array of elements
     * @param key   searched key value
     * @return index of element equal to key. If no such element exists, returns index of the first element
     * greater than key or the last element in the array
     */
    private int search(@Nonnull double[] array, double key) {
        int index = Arrays.binarySearch(array, key);
        if (index < 0) index = -index - 1;
        if (index >= array.length) index = array.length - 1;
        return index;
    }

    private double getX(@Nonnull Chromatogram c, int i, int j, double y) {
        return c.xs[i] + (c.xs[j] - c.xs[i]) / (c.ys[j] - c.ys[i]) * (y - c.ys[i]);
    }
}
