/*
 * Copyright (C) 2017 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.workflow.decomposition;

import com.google.common.collect.Range;
import dulab.adap.datamodel.Chromatogram;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.IntStream;

/**
 * Detects peaks of a chromatogram.
 *
 * The algorithm is similar to MS-DIAL algorithm, which detects peaks based on the values of the first and second
 * derivatives.
 *
 * @author Du-Lab Team dulab.binf@gmail.com
 */
public class PeakDetector {

    /* Coefficients used for estimating the noise level */
    private final static double AMPLITUDE_NOISE_FACTOR = 4.0;
    private final static double SLOPE_NOISE_FACTOR = 2.0;
    private final static double PEAK_TOP_NOISE_FACTOR = 2.0;

    /* Coefficients used for estimating the first and second derivatives */
    private final static double[] FIRST_DERIVATIVE_COEFFICIENTS = new double[] { -0.2, -0.1, 0, 0.1, 0.2 };
    private final static double[] SECOND_DERIVATIVE_COEFFICIENTS = new double[]
            {0.14285714, -0.07142857, -0.1428571, -0.07142857, 0.14285714};
    private final static int HALF_NUM_COEFFICIENTS = FIRST_DERIVATIVE_COEFFICIENTS.length / 2;

    /* Number of smoothing points */
    private final int numSmoothingPoints;

    /* Minimum height of detected peaks */
    private final double minAmplitude;

    /* Minimum and maximum duration of detected peaks */
    private final Range<Double> durationRange;

    /* Different states of peak shape */
    private enum PeakPhase {ASCENT, TOP, DESCENT, END}

    /**
     * Creates an instance of {@link PeakDetector}.
     * @param numSmoothingPoints number of smoothing points
     * @param minAmplitude minimum height of detected peaks
     * @param durationRange minimum and maximum duration of detected peaks
     */
    public PeakDetector(int numSmoothingPoints, double minAmplitude, @Nonnull Range<Double> durationRange)
    {
        this.numSmoothingPoints = numSmoothingPoints;
        this.minAmplitude = minAmplitude;
        this.durationRange = durationRange;
    }

    /**
     * Detects peaks of a chromatogram.
     * @param chromatogram an instance of {@link Chromatogram}
     * @param mz mz-value associated with the chromatogram
     * @return list of {@link RetTimeClusterer.Item} containing ranges of the detected peaks
     */
    public List<RetTimeClusterer.Interval> run(@Nonnull Chromatogram chromatogram, double mz)
    {
        final int numPoints = chromatogram.length;

        // Copy chromatograms values so that the chromatogram will no change after the peak detection.
        final double[] retTimes = chromatogram.xs.clone();
        final double[] intensities = chromatogram.ys.clone();

        smoothArray(intensities, numSmoothingPoints);

        // ----------------------------------------------
        // Estimation of the first and second derivatives
        // ----------------------------------------------

        double[] firstDerivatives = new double[numPoints];
        double[] secondDerivatives = new double[numPoints];
        for (int i = HALF_NUM_COEFFICIENTS; i < numPoints - HALF_NUM_COEFFICIENTS; ++i) {
            for (int j = 0; j < FIRST_DERIVATIVE_COEFFICIENTS.length; ++j) {
                firstDerivatives[i] += FIRST_DERIVATIVE_COEFFICIENTS[j] * intensities[i + j - HALF_NUM_COEFFICIENTS];
                secondDerivatives[i] += SECOND_DERIVATIVE_COEFFICIENTS[j] * intensities[i + j - HALF_NUM_COEFFICIENTS];
            }
        }

        double maxFirstDifference = Arrays.stream(firstDerivatives).map(Math::abs).max().orElse(0.0);
        double maxSecondDifference = -Arrays.stream(secondDerivatives).filter(x -> x < 0.0).min().orElse(0.0);
        double maxAmplitudeDifference = IntStream.range(1, intensities.length)
                .mapToDouble(i -> Math.abs(intensities[i] - intensities[i - 1])).max().orElse(0.0);

        // --------------------------
        // Calculating the thresholds
        // --------------------------

        Noise noise = new Noise(intensities, firstDerivatives, secondDerivatives,
                new NoiseThresholds(maxAmplitudeDifference, maxFirstDifference, maxSecondDifference));

        // --------------------
        // Start peak detection
        // --------------------

        List<RetTimeClusterer.Interval> peakRanges = new ArrayList<>();
        for (int i = 0; i < numPoints; ++i)
        {
            if (i >= numPoints - 1) break;  //  - minDataPoints

            int peakStart;
            int peakEnd;

            if (firstDerivatives[i] > noise.slope && firstDerivatives[i + 1] > noise.slope)
            {
                // Start a new peak

                peakStart = i;

                // Correct the left edge if necessary
                for (int j = 0; j <= 5; ++j) {
                    if (i - j - 1 < 0) break;
                    if (intensities[i - j] <= intensities[i - j - 1]) break;
                    if (intensities[i - j] > intensities[i - j - 1]) peakStart = i - j - 1;
                }

                // Detect the top and the right edge of the peak
                PeakPhase peakPhase = PeakPhase.ASCENT;
                while (++i < numPoints - 1 && peakPhase != PeakPhase.END)
                {
                    switch (peakPhase) {
                        case ASCENT:
                            if (firstDerivatives[i - 1] > 0.0 && firstDerivatives[i] < 0.0
                                    && secondDerivatives[i] < -noise.peakTop)
                                peakPhase = PeakPhase.TOP;
                            break;
                        case TOP:
                            if (firstDerivatives[i - 1] < -noise.slope && firstDerivatives[i] < -noise.slope)
                                peakPhase = PeakPhase.DESCENT;
                            break;
                        case DESCENT:
                            if (firstDerivatives[i - 1] > -noise.slope && firstDerivatives[i] > -noise.slope)
                                peakPhase = PeakPhase.END;
                            break;
                    }
                }

                peakEnd = i;

                // Correct the right edge of the peak
                // Case: wrong edge is on the left of real edge
                boolean rightCheck = false;
                for (int j = 0; j <= 5; ++j) {
                    if (i + j + 1 > numPoints - 1) break;
                    if (intensities[i + j] <= intensities[i + j + 1]) break;
                    if (intensities[i + j] > intensities[i + j + 1]) {
                        rightCheck = true;
                        peakEnd = i + j + 1;
                    }
                }
                // Case: wrong edge is on the right of real edge
                if (!rightCheck) {
                    for (int j = 0; j <= 5; ++j) {
                        if (i - j - 1 < 0) break;
                        if (intensities[i - j] <= intensities[i - j - 1]) break;
                        if (intensities[i - j] > intensities[i - j - 1]) {
                            peakEnd = i - j - 1;
                        }
                    }
                }

                // Check peak criteria
                double peakApexIntensity = Double.MIN_VALUE;
                int peakApex = peakStart;
                for (int j = peakStart; j <= peakEnd; ++j)
                    if (peakApexIntensity < intensities[j]) {
                        peakApexIntensity = intensities[j];
                        peakApex = j;
                    }
                if (intensities[peakApex] - intensities[peakStart] < Math.max(minAmplitude, noise.amplitude)
                        || intensities[peakApex] - intensities[peakEnd] < Math.max(minAmplitude, noise.amplitude))
                    continue;

                // Find the range of intensities above 5% of the maximum
                int peakAccurateStart = peakApex;
                while (peakAccurateStart > peakStart && intensities[peakAccurateStart] > 0.05 * intensities[peakApex])
                    --peakAccurateStart;

                int peakAccurateEnd = peakApex;
                while (peakAccurateEnd < peakEnd && intensities[peakAccurateEnd] > 0.05 * intensities[peakApex])
                    ++peakAccurateEnd;

                // Check peak duration
                if (!durationRange.contains(retTimes[peakAccurateEnd] - retTimes[peakAccurateStart]))
                    continue;

                peakRanges.add(new RetTimeClusterer.Interval(
                        Range.closed(retTimes[peakAccurateStart], retTimes[peakAccurateEnd]), mz));
            }
        }
        return peakRanges;
    }

    /**
     * Performs smoothing of the data suing the weighted moving average algorithm.
     * @param array arrays of values
     * @param n number of smoothing points (positive integer)
     */
    private void smoothArray(@Nonnull double[] array, int n)
    {
        final int denominator = n * n;
        double[] result = IntStream.range(n - 1, array.length - n + 1).mapToDouble(i ->
                (n * array[i] + IntStream.range(1, n).mapToDouble(j ->
                        j * (array[i - n + j] + array[i + n - j])).sum()) / denominator).toArray();

        System.arraycopy(result, 0, array, n - 1, result.length);
    }

    private static class Noise
    {
        double amplitude;
        double slope;
        double peakTop;

        Noise(double[] intensities, double[] firstDerivatives, double[] secondDerivatives,
              NoiseThresholds thresholds)
        {
            final int numPoints = intensities.length;

            List<Double> amplitudeNoiseCandidates = new ArrayList<>();
            List<Double> slopeNoiseCandidates = new ArrayList<>();
            List<Double> peakTopNoiseCandidates = new ArrayList<>();

            for (int i = 2; i < numPoints - 2; ++i)
            {
                double amplitudeDifference = Math.abs(intensities[i + 1] - intensities[i]);
                if (amplitudeDifference < thresholds.amplitude && amplitudeDifference > 0.0)
                    amplitudeNoiseCandidates.add(amplitudeDifference);

                if (Math.abs(firstDerivatives[i]) < thresholds.slope && Math.abs(firstDerivatives[i]) > 0.0)
                    slopeNoiseCandidates.add(Math.abs(firstDerivatives[i]));

                if (secondDerivatives[i] < 0.0 && Math.abs(secondDerivatives[i]) < thresholds.peakTop)
                    peakTopNoiseCandidates.add(Math.abs(secondDerivatives[i]));
            }

            Collections.sort(amplitudeNoiseCandidates);
            Collections.sort(slopeNoiseCandidates);
            Collections.sort(peakTopNoiseCandidates);

            amplitude = amplitudeNoiseCandidates.isEmpty() ?
                    0.0001 :
                    amplitudeNoiseCandidates.get(amplitudeNoiseCandidates.size() / 2);
            slope = slopeNoiseCandidates.isEmpty() ?
                    0.0001 :
                    slopeNoiseCandidates.get(slopeNoiseCandidates.size() / 2);
            peakTop = peakTopNoiseCandidates.isEmpty() ?
                    0.0001 :
                    peakTopNoiseCandidates.get(peakTopNoiseCandidates.size() / 2);

            amplitude *= AMPLITUDE_NOISE_FACTOR;
            slope *= SLOPE_NOISE_FACTOR;
            peakTop *= PEAK_TOP_NOISE_FACTOR;
        }
    }

    private static class NoiseThresholds {
        double amplitude;
        double slope;
        double peakTop;

        NoiseThresholds(double maxAmplitudeDifference, double maxFirstDifference, double maxSecondDifference) {
            amplitude = 0.05 * maxAmplitudeDifference;
            slope = 0.05 * maxFirstDifference;
            peakTop = 0.05 * maxSecondDifference;
        }
    }
}
