/*
 * Copyright (C) 2018 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.workflow.decomposition;

import dulab.adap.datamodel.BetterPeak;
import dulab.adap.datamodel.Chromatogram;

import java.util.*;
import java.util.stream.Collectors;

public class PeakList extends ArrayList<Peak> {

    /**
     * Creates a list of {@link ComponentSelector.Peak}. Each {@link ComponentSelector.Peak} corresponds to one range and have a chromatogram with
     * the specified retention times only.
     *
     * @param chromatograms list of all chromatograms
     * @param peaks         list of peaks used to filter chromatograms
     * @param retTimes      sorted array of retentions times used to construct new short chromatograms
     */
    public PeakList(List<BetterPeak> chromatograms,
                    List<BetterPeak> detectedPeaks,
                    double[] retTimes,
                    boolean adjustApexRetTime) {

        super(detectedPeaks.size());

        for (BetterPeak detectedPeak : detectedPeaks) {

            double[] intensities;
            if (detectedPeak.getParentId() != null) {

                BetterPeak c = chromatograms.stream()
                        .filter(chromatogram -> chromatogram.getId() == detectedPeak.getParentId())
                        .filter(chromatogram -> chromatogram.getFirstRetTime() <= detectedPeak.getRetTime())
                        .filter(chromatogram -> detectedPeak.getRetTime() <= chromatogram.getLastRetTime())
                        .findFirst()
                        .orElseThrow(() -> new IllegalArgumentException("Inconsistency of m/z values in chromatograms and ranges"));

                intensities = getIntensities(c.chromatogram, retTimes);
            }
            else {
                intensities = getIntensities(detectedPeak.chromatogram, retTimes);
            }

            Peak peak = new Peak(
                    new Chromatogram(retTimes, intensities),
                    detectedPeak.getFirstRetTime(),
                    detectedPeak.getLastRetTime(),
                    detectedPeak.getMZ(),
                    adjustApexRetTime);

            if (peak.chromatogram.norm > 0.0)
                this.add(peak);
        }
    }

    /**
     * Creates a list of {@link ComponentSelector.Peak}. Each {@link ComponentSelector.Peak} have a chromatogram with the specified retention times only.
     *
     * @param chromatograms list of chromatograms
     * @param retTimes      sorted array of retention times used to construct short chromatograms
     */
    public PeakList(List<BetterPeak> chromatograms, double[] retTimes, boolean adjustApexRetTime) {

        super();

        if (retTimes.length == 0) return;

        double startRetTime = retTimes[0];
        double endRetTime = retTimes[retTimes.length - 1];

        for (BetterPeak chromatogram : chromatograms) {
            double[] intensities = getIntensities(chromatogram.chromatogram, retTimes);
            this.add(new Peak(
                    new Chromatogram(retTimes, intensities),
                    startRetTime,
                    endRetTime,
                    chromatogram.getMZ(),
                    adjustApexRetTime));
        }
    }

    private double[] getIntensities(Chromatogram chromatogram, double[] retTimes) {

        double[] intensities = new double[retTimes.length];
        for (int i = 0; i < retTimes.length; ++i) {
            int index = Arrays.binarySearch(chromatogram.xs, retTimes[i]);
            if (index >= 0)
                intensities[i] = chromatogram.ys[index];
            else
                intensities[i] = interpolateY(chromatogram, -index - 2, -index - 1, retTimes[i]);
        }
        return intensities;
    }

    private double interpolateY(Chromatogram c, int ind1, int ind2, double t)
            throws IndexOutOfBoundsException {

        boolean index1OutOfBounds = ind1 < 0 || ind1 >= c.length;
        boolean index2OutOfBounds = ind2 < 0 || ind2 >= c.length;

        if (index1OutOfBounds && index2OutOfBounds)
            throw new IndexOutOfBoundsException();

        else if (index1OutOfBounds)
            return c.ys[ind2];

        else if (index2OutOfBounds)
            return c.ys[ind1];

        return c.ys[ind1] + (c.ys[ind2] - c.ys[ind1]) / (c.xs[ind2] - c.xs[ind1]) * (t - c.xs[ind1]);
    }
}
