/*
 * Copyright (C) 2018 Du-Lab Team <dulab.binf@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dulab.adap.workflow.peakannotation;

import dulab.adap.datamodel.Adduct;
import dulab.adap.datamodel.Spectrum;
import dulab.adap.workflow.peakannotation.rules.AdductList;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.IntStream;

public class PeakAnnotation {

    private double processed = 0.0;
    private boolean canceled = false;

    public double getProcessedPercent() {
        return processed;
    }

    public void cancel() {
        canceled = true;
    }

    public Optional<Double> run(Spectrum spectrum, Parameters params) throws IllegalStateException {

        final double height = Arrays
                .stream(spectrum.ys)
                .max()
                .orElseThrow(() -> new IllegalStateException("Cannot find height of the spectrum."));

        // Find all peaks above 10%
        final double cutOff = params.intensityThreshold * height;
        int[] peakIndices = IntStream
                .range(0, spectrum.length)
                .filter(i -> spectrum.ys[i] > cutOff)
                .toArray();

        if (peakIndices.length == 0)
            throw new IllegalStateException("Array of peaks is empty.");

        // Estimate evidence that m/z is the precursor mass by calculating the number of explained peaks.
        final double processedStep = 1.0 / peakIndices.length;
        processed = 0.0;

        int bestCount = 0;
        int bestCountIndex = peakIndices[0];
        for (int i : peakIndices) {

            if (canceled) return Optional.empty();

            double molecularMass = getMolecularMass(spectrum.getMZ(i), params.hydrogenAdduct);

            int countExplainedPeaks = 0;
            for (double mz : spectrum.xs) {

                boolean isExplained = params.otherAdducts
                        .stream()
                        .mapToDouble(a -> getMolecularMass(mz, a) - molecularMass)
                        .map(Math::abs)
                        .filter(d -> d < params.mzTolerance)
                        .findAny()
                        .isPresent();

                if (isExplained) countExplainedPeaks += 1;
            }

            if (countExplainedPeaks > bestCount || (countExplainedPeaks == bestCount
                    && spectrum.getIntensity(i) > spectrum.getIntensity(bestCountIndex))) {
                bestCount = countExplainedPeaks;
                bestCountIndex = i;
            }

            processed += processedStep;
        }

        return Optional.of(spectrum.getMZ(bestCountIndex));
    }

    private double getMolecularMass(double mz, Adduct adduct) {
        return (adduct.getCharge() * mz - adduct.getMass()) / adduct.getNumMolecules();
    }


    public static class Parameters {

        public final double intensityThreshold;
        public final double mzTolerance;
        public final Adduct hydrogenAdduct;
        public final List<Adduct> otherAdducts;

        public Parameters(double intensityThreshold, double mzTolerance, Adduct hydrogenAdduct, List<Adduct> otherAdducts) {
            this.intensityThreshold = intensityThreshold;
            this.mzTolerance = mzTolerance;
            this.hydrogenAdduct = Objects.requireNonNull(hydrogenAdduct, "Hydrogen adduct is required.");
            this.otherAdducts = Objects.requireNonNull(otherAdducts, "List of other adducts is required.");
        }

        public Parameters(double intensityThreshold, double mzTolerance, AdductList adductList) {

            this.intensityThreshold = intensityThreshold;
            this.mzTolerance = mzTolerance;

            List<Adduct> adducts = adductList.get();
            this.hydrogenAdduct = adducts.get(0);
            this.otherAdducts = adducts.subList(1, adducts.size());
        }
    }
}
