package org.duelengine.duel.mvc;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Sets cache control to "never" expire & enables cross-origin access.
 *
 * Only use for SHA1-named CDN resources which change name as content changes.
 * 
 * http://www.w3.org/Protocols/rfc2616/rfc2616-sec13.html
 * 
 * To mark a response as "never expires," an origin server sends an
 * Expires date approximately one year from the time the response is sent.
 * HTTP/1.1 servers SHOULD NOT send Expires dates more than one year in the future.
 * 
 * CDN-based resources require permission to be accessed from another domain.
 * e.g. without, CSS references to other resources like fonts may be blocked
 *
 * http://www.w3.org/TR/cors/#access-control-allow-origin-response-hea
 * https://developer.mozilla.org/En/HTTP_access_control#Access-Control-Allow-Origin
 */
class CrossOriginResourceSharingFilter implements Filter {
	private final String hostRegExp;
	private final boolean withCredentials;

	public CrossOriginResourceSharingFilter(String hostRegExp, boolean withCredentials) {
		this.hostRegExp = (hostRegExp == null || hostRegExp.equals("*")) ? null : hostRegExp;
		this.withCredentials = withCredentials;
	}

	public void init(FilterConfig config) {}

	public void destroy() {}

	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
			throws IOException, ServletException {

		if (request instanceof HttpServletRequest && response instanceof HttpServletResponse) {
			doFilter((HttpServletRequest)request, (HttpServletResponse)response, chain);

		} else {
			chain.doFilter(request, response);
		}
	}

	private void doFilter(HttpServletRequest request, HttpServletResponse response, FilterChain chain)
			throws IOException, ServletException {

		// TODO: limit to only valid domains
		// TODO: also verify "Host" header for DNS spoofing
		// http://www.w3.org/TR/cors/#security
		String origin = request.getHeader("Origin");

		if (origin == null || origin.isEmpty()) {
			// not a CORS request
			chain.doFilter(request, response);
			return;
		}

		if (!origin.matches(hostRegExp)) {
			// shortcut blocked CORS request
			response.setStatus(HttpServletResponse.SC_FORBIDDEN);
			return;
		}

		response.addHeader("Access-Control-Allow-Origin", (hostRegExp == null) ? "*" : origin);
		if (withCredentials) {
			response.addHeader("Access-Control-Allow-Credentials", "true");
		}

		if (request.getHeader("Access-Control-Request-Method") != null &&
			"OPTIONS".equalsIgnoreCase(request.getMethod())) {

			// CORS preflight request
			response.addHeader("Access-Control-Allow-Methods", "GET,POST,PUT");
			response.addHeader("Access-Control-Allow-Headers", "Cookie");
			response.addHeader("Access-Control-Max-Age", "360");

			// shortcut result
			response.setStatus(HttpServletResponse.SC_OK);
		}

		response.addHeader("Access-Control-Expose-Headers", "Cookie");

		chain.doFilter(request, response);
	}
}
