/*
 * Decompiled with CFR 0.152.
 */
package org.duracloud.common.util;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;
import javax.sql.DataSource;
import org.apache.derby.jdbc.EmbeddedDataSource;
import org.duracloud.common.model.Credential;
import org.duracloud.common.util.TableSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcOperations;
import org.springframework.jdbc.core.JdbcTemplate;

public abstract class DatabaseUtil {
    protected final Logger log = LoggerFactory.getLogger(DatabaseUtil.class);
    private final EmbeddedDataSource dataSource;
    private final JdbcTemplate jdbcTemplate;
    private final String baseDir;
    public static String NOT_ENCRYPTED = "NOT_ENCRYPTED";

    public DatabaseUtil(Credential cred, String baseDir) {
        this(cred, baseDir, NOT_ENCRYPTED);
    }

    public DatabaseUtil(Credential cred, String baseDir, String bootPassword) {
        this.baseDir = baseDir;
        this.dataSource = new EmbeddedDataSource();
        this.dataSource.setUser(cred.getUsername());
        this.dataSource.setPassword(cred.getPassword());
        this.dataSource.setDatabaseName(baseDir);
        if (!bootPassword.equals(NOT_ENCRYPTED)) {
            String connAtts = "dataEncryption=true;bootPassword=" + bootPassword;
            this.dataSource.setConnectionAttributes(connAtts);
        }
        this.jdbcTemplate = new JdbcTemplate((DataSource)this.dataSource);
    }

    protected abstract List<TableSpec> getTableSpecs();

    public void initializeDB() throws Exception {
        this.log.debug("initializing db");
        this.ensureDatabaseExists();
        this.ensureTablesExist();
        this.ensureTablesCleared();
    }

    public void ensureDatabaseExists() {
        Connection conn = null;
        try {
            conn = this.dataSource.getConnection();
        }
        catch (SQLException e) {
            this.log.info("creating database... " + this.baseDir);
            this.dataSource.setCreateDatabase("create");
        }
        finally {
            this.close(conn);
        }
    }

    public void ensureTablesExist() throws SQLException {
        for (TableSpec ts : this.getTableSpecs()) {
            try {
                this.execute("SELECT " + ts.getPrimaryKey() + " FROM " + ts.getTableName());
            }
            catch (Exception e) {
                this.log.info("creating table... " + ts.getTableName());
                this.execute(ts.getDdl());
            }
        }
        Connection conn = null;
        try {
            conn = this.dataSource.getConnection();
            DatabaseMetaData metadata = conn.getMetaData();
            ResultSet results = metadata.getTables(null, null, "*", null);
            while (results.next()) {
                this.log.info("examining tables:" + results.getObject(0).toString());
            }
        }
        catch (SQLException e) {
            this.log.error("Database should already exist.");
            throw e;
        }
        finally {
            this.close(conn);
        }
    }

    public void clearDB() {
        this.clearTables();
    }

    private void ensureTablesCleared() {
        this.clearTables();
    }

    private void clearTables() {
        LinkedList<TableSpec> reversedSpecs = new LinkedList<TableSpec>();
        for (TableSpec spec : this.getTableSpecs()) {
            reversedSpecs.addFirst(spec);
        }
        for (TableSpec spec : reversedSpecs) {
            this.execute("DELETE FROM " + spec.getTableName() + " WHERE " + spec.getPrimaryKey() + " IS NOT NULL");
        }
    }

    private void close(Connection conn) {
        if (conn != null) {
            try {
                conn.close();
            }
            catch (SQLException e) {
                this.log.warn("Exception closing DB connection: " + e.getMessage());
            }
        }
    }

    public void disconnect() {
        this.dataSource.setShutdownDatabase("shutdown");
    }

    public JdbcTemplate getSimpleJdbcTemplate() {
        return this.jdbcTemplate;
    }

    public DataSource getDataSource() {
        return this.dataSource;
    }

    public JdbcOperations getOps() {
        return this.getSimpleJdbcTemplate();
    }

    protected void execute(String sql) {
        this.getOps().execute(sql);
    }

    private void dropTables() {
        for (TableSpec ts : this.getTableSpecs()) {
            this.execute("DROP TABLE " + ts.getTableName());
        }
    }

    private void deleteDB(String baseDir) throws IOException {
        File base = new File(baseDir);
        this.deleteFiles(base);
    }

    private void deleteFiles(File file) {
        if (file.isDirectory()) {
            for (File f : file.listFiles()) {
                this.deleteFiles(f);
            }
        }
        file.delete();
    }
}

