/*
 * Decompiled with CFR 0.152.
 */
package org.duracloud.retrieval.mgmt;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.FileTime;
import java.text.ParseException;
import java.util.Date;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.duracloud.chunk.util.ChunkUtil;
import org.duracloud.common.model.ContentItem;
import org.duracloud.common.util.ChecksumUtil;
import org.duracloud.common.util.DateUtil;
import org.duracloud.retrieval.mgmt.OutputWriter;
import org.duracloud.retrieval.mgmt.StatusManager;
import org.duracloud.retrieval.source.ContentStream;
import org.duracloud.retrieval.source.RetrievalSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetrievalWorker
implements Runnable {
    private final Logger logger = LoggerFactory.getLogger(RetrievalWorker.class);
    private static final int MAX_ATTEMPTS = 5;
    private static final String COPY = "-copy";
    private ContentItem contentItem;
    private RetrievalSource source;
    private File contentDir;
    private boolean overwrite;
    private OutputWriter outWriter;
    private boolean createSpaceDir;
    private boolean applyTimestamps;
    private int attempts;
    private File localFile;
    private ContentStream contentStream;
    private StatusManager statusManager;

    public RetrievalWorker(ContentItem contentItem, RetrievalSource source, File contentDir, boolean overwrite, OutputWriter outWriter, boolean createSpaceDir, boolean applyTimestamps) {
        this.contentItem = contentItem;
        this.source = source;
        this.contentDir = contentDir;
        this.overwrite = overwrite;
        this.outWriter = outWriter;
        this.createSpaceDir = createSpaceDir;
        this.applyTimestamps = applyTimestamps;
        this.statusManager = StatusManager.getInstance();
        this.attempts = 0;
    }

    @Override
    public void run() {
        this.statusManager.startingWork();
        this.retrieveFile();
    }

    public Map<String, String> retrieveFile() {
        ++this.attempts;
        File localFile = this.getLocalFile();
        Map<String, String> props = null;
        try {
            if (localFile.exists()) {
                props = this.getContentProperties();
                if (this.checksumsMatch(localFile, props.get("content-checksum"))) {
                    this.noChangeNeeded(localFile.getAbsolutePath());
                } else {
                    if (this.overwrite) {
                        this.deleteFile(localFile);
                    } else {
                        this.renameFile(localFile);
                    }
                    props = this.retrieveToFile(localFile);
                    this.succeed(localFile.getAbsolutePath());
                }
            } else {
                File parentDir = localFile.getParentFile();
                if (!parentDir.exists()) {
                    parentDir.mkdirs();
                    parentDir.setWritable(true);
                }
                props = this.retrieveToFile(localFile);
                this.succeed(localFile.getAbsolutePath());
            }
        }
        catch (Exception e) {
            this.logger.error("Exception retrieving remote file " + this.contentItem.getContentId() + " as local file " + localFile.getAbsolutePath() + ": " + e.getMessage(), (Throwable)e);
            if (this.attempts < 5) {
                props = this.retrieveFile();
            }
            this.fail(e.getMessage());
        }
        return props;
    }

    public File getLocalFile() {
        if (this.localFile == null) {
            ChunkUtil util = new ChunkUtil();
            String contentId = this.contentItem.getContentId();
            if (util.isChunkManifest(contentId)) {
                contentId = util.preChunkedContentId(contentId);
            }
            if (this.createSpaceDir) {
                File spaceDir = new File(this.contentDir, this.contentItem.getSpaceId());
                this.localFile = new File(spaceDir, contentId);
            } else {
                this.localFile = new File(this.contentDir, contentId);
            }
        }
        return this.localFile;
    }

    protected boolean checksumsMatch(File localFile) throws IOException {
        return this.checksumsMatch(localFile, null);
    }

    protected boolean checksumsMatch(File localFile, String remoteChecksum) throws IOException {
        if (remoteChecksum == null || "".equals(remoteChecksum)) {
            remoteChecksum = this.contentStream != null ? this.contentStream.getChecksum() : this.source.getSourceChecksum(this.contentItem);
        }
        String localChecksum = this.getChecksum(localFile);
        return localChecksum.equals(remoteChecksum);
    }

    protected String getChecksum(File localFile) throws IOException {
        ChecksumUtil checksumUtil = new ChecksumUtil(ChecksumUtil.Algorithm.MD5);
        String localChecksum = checksumUtil.generateChecksum(localFile);
        return localChecksum;
    }

    protected File renameFile(File localFile) throws IOException {
        File origFile = new File(localFile.getAbsolutePath());
        File copiedFile = new File(localFile.getParent(), localFile.getName() + COPY);
        int i = 2;
        while (copiedFile.exists()) {
            copiedFile = new File(localFile.getParent(), localFile.getName() + COPY + "-" + i);
            ++i;
        }
        FileUtils.moveFile((File)origFile, (File)copiedFile);
        return copiedFile;
    }

    protected void deleteFile(File localFile) throws IOException {
        localFile.delete();
    }

    protected Map<String, String> getContentProperties() {
        Map<String, String> properties = null;
        properties = this.contentStream != null ? this.contentStream.getProperties() : this.source.getSourceProperties(this.contentItem);
        return properties;
    }

    protected Map<String, String> retrieveToFile(File localFile) throws IOException {
        this.contentStream = this.source.getSourceContent(this.contentItem);
        try (InputStream inStream = this.contentStream.getStream();
             FileOutputStream outStream = new FileOutputStream(localFile);){
            IOUtils.copyLarge((InputStream)inStream, (OutputStream)outStream);
        }
        catch (IOException e) {
            try {
                this.deleteFile(localFile);
            }
            catch (IOException ioe) {
                this.logger.error("Exception deleting local file " + localFile.getAbsolutePath() + " due to: " + ioe.getMessage());
            }
            throw e;
        }
        if (!this.checksumsMatch(localFile, this.contentStream.getChecksum())) {
            this.deleteFile(localFile);
            throw new IOException("Calculated checksum value for retrieved file does not match properties checksum.");
        }
        if (this.applyTimestamps) {
            this.applyTimestamps(this.contentStream, localFile);
        }
        return this.contentStream.getProperties();
    }

    protected void applyTimestamps(ContentStream content, File localFile) {
        FileTime createTime = this.convertDateToFileTime(content.getDateCreated());
        FileTime lastAccessTime = this.convertDateToFileTime(content.getDateLastAccessed());
        FileTime lastModTime = this.convertDateToFileTime(content.getDateLastModified());
        BasicFileAttributeView fileAttributeView = Files.getFileAttributeView(localFile.toPath(), BasicFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
        try {
            fileAttributeView.setTimes(lastModTime, lastAccessTime, createTime);
        }
        catch (IOException e) {
            this.logger.error("Error setting timestamps for local file " + localFile.getAbsolutePath() + ": " + e.getMessage(), (Throwable)e);
        }
    }

    private FileTime convertDateToFileTime(String strDate) {
        FileTime time = null;
        if (null != strDate) {
            Date date = null;
            try {
                date = DateUtil.convertToDate((String)strDate, (DateUtil.DateFormat)DateUtil.DateFormat.LONG_FORMAT);
            }
            catch (ParseException e) {
                date = null;
            }
            if (null != date) {
                time = FileTime.fromMillis(date.getTime());
            }
        }
        return time;
    }

    protected void noChangeNeeded(String localFilePath) {
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Local file " + localFilePath + " matches remote file " + this.contentItem.toString() + " no update needed");
        }
        this.statusManager.noChangeCompletion();
    }

    protected void succeed(String localFilePath) {
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Successfully retrieved " + this.contentItem.toString() + " to local file " + localFilePath);
        }
        this.outWriter.writeSuccess(this.contentItem, localFilePath, this.attempts);
        this.statusManager.successfulCompletion();
    }

    protected void fail(String errMsg) {
        String error = "Failed to retrieve " + this.contentItem.toString() + " after " + this.attempts + " attempts. Last error message was: " + errMsg;
        this.logger.error(error);
        System.err.println(error);
        this.outWriter.writeFailure(this.contentItem, error, this.attempts);
        this.statusManager.failedCompletion();
    }
}

