/*
 * Decompiled with CFR 0.152.
 */
package org.duracloud.s3storage;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.AccessControlList;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.Bucket;
import com.amazonaws.services.s3.model.BucketLifecycleConfiguration;
import com.amazonaws.services.s3.model.BucketTaggingConfiguration;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.CopyObjectResult;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.StorageClass;
import com.amazonaws.services.s3.model.TagSet;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.duracloud.common.stream.ChecksumInputStream;
import org.duracloud.common.util.ChecksumUtil;
import org.duracloud.common.util.DateUtil;
import org.duracloud.s3storage.S3ProviderUtil;
import org.duracloud.s3storage.StoragePolicy;
import org.duracloud.storage.domain.ContentByteRange;
import org.duracloud.storage.domain.ContentIterator;
import org.duracloud.storage.domain.RetrievedContent;
import org.duracloud.storage.domain.StorageProviderType;
import org.duracloud.storage.error.ChecksumMismatchException;
import org.duracloud.storage.error.NotFoundException;
import org.duracloud.storage.error.StorageException;
import org.duracloud.storage.provider.StorageProvider;
import org.duracloud.storage.provider.StorageProviderBase;
import org.duracloud.storage.util.StorageProviderUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3StorageProvider
extends StorageProviderBase {
    private final Logger log = LoggerFactory.getLogger(S3StorageProvider.class);
    protected static final int MAX_ITEM_COUNT = 1000;
    private static final StorageClass DEFAULT_STORAGE_CLASS = StorageClass.Standard;
    private static final String UTF_8 = StandardCharsets.UTF_8.name();
    protected static final String HIDDEN_SPACE_PREFIX = "hidden-";
    protected static final String HEADER_VALUE_PREFIX = UTF_8 + "''";
    protected static final String HEADER_KEY_SUFFIX = "*";
    private String accessKeyId = null;
    protected AmazonS3Client s3Client = null;

    public S3StorageProvider(String accessKey, String secretKey) {
        this(S3ProviderUtil.getAmazonS3Client(accessKey, secretKey, null), accessKey, null);
    }

    public S3StorageProvider(String accessKey, String secretKey, Map<String, String> options) {
        this(S3ProviderUtil.getAmazonS3Client(accessKey, secretKey, options), accessKey, options);
    }

    public S3StorageProvider(AmazonS3Client s3Client, String accessKey, Map<String, String> options) {
        this.accessKeyId = accessKey;
        this.s3Client = s3Client;
    }

    public StorageProviderType getStorageProviderType() {
        return StorageProviderType.AMAZON_S3;
    }

    public Iterator<String> getSpaces() {
        this.log.debug("getSpaces()");
        ArrayList<String> spaces = new ArrayList<String>();
        List<Bucket> buckets = this.listAllBuckets();
        for (Bucket bucket : buckets) {
            String bucketName = bucket.getName();
            if (!this.isSpace(bucketName)) continue;
            spaces.add(this.getSpaceId(bucketName));
        }
        Collections.sort(spaces);
        return spaces.iterator();
    }

    private List<Bucket> listAllBuckets() {
        try {
            return this.s3Client.listBuckets();
        }
        catch (AmazonClientException e) {
            String err = "Could not retrieve list of S3 buckets due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    public Iterator<String> getSpaceContents(String spaceId, String prefix) {
        this.log.debug("getSpaceContents(" + spaceId + ", " + prefix);
        this.throwIfSpaceNotExist(spaceId);
        return new ContentIterator((StorageProvider)this, spaceId, prefix);
    }

    public List<String> getSpaceContentsChunked(String spaceId, String prefix, long maxResults, String marker) {
        this.log.debug("getSpaceContentsChunked(" + spaceId + ", " + prefix + ", " + maxResults + ", " + marker + ")");
        String bucketName = this.getBucketName(spaceId);
        if (maxResults <= 0L) {
            maxResults = 10000L;
        }
        return this.getCompleteBucketContents(bucketName, prefix, maxResults, marker);
    }

    private List<String> getCompleteBucketContents(String bucketName, String prefix, long maxResults, String marker) {
        ArrayList<String> contentItems = new ArrayList<String>();
        List<S3ObjectSummary> objects = this.listObjects(bucketName, prefix, maxResults, marker);
        for (S3ObjectSummary object : objects) {
            contentItems.add(object.getKey());
        }
        return contentItems;
    }

    private List<S3ObjectSummary> listObjects(String bucketName, String prefix, long maxResults, String marker) {
        int numResults = new Long(maxResults).intValue();
        ListObjectsRequest request = new ListObjectsRequest(bucketName, prefix, marker, null, Integer.valueOf(numResults));
        try {
            ObjectListing objectListing = this.s3Client.listObjects(request);
            return objectListing.getObjectSummaries();
        }
        catch (AmazonClientException e) {
            String err = "Could not get contents of S3 bucket " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    protected boolean spaceExists(String spaceId) {
        try {
            this.getBucketName(spaceId);
            return true;
        }
        catch (NotFoundException e) {
            return false;
        }
    }

    public void createSpace(String spaceId) {
        this.log.debug("createSpace(" + spaceId + ")");
        this.throwIfSpaceExists(spaceId);
        Bucket bucket = this.createBucket(spaceId);
        Date created = bucket.getCreationDate();
        if (created == null) {
            created = new Date();
        }
        HashMap spaceACLs = new HashMap();
        HashMap<String, String> spaceProperties = new HashMap<String, String>();
        spaceProperties.put("space-created", this.formattedDate(created));
        try {
            this.setNewSpaceProperties(spaceId, spaceProperties, spaceACLs);
        }
        catch (StorageException e) {
            this.removeSpace(spaceId);
            String err = "Unable to create space due to: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    private Bucket createBucket(String spaceId) {
        String bucketName = this.getNewBucketName(spaceId);
        try {
            Bucket bucket = this.s3Client.createBucket(bucketName);
            StoragePolicy storagePolicy = this.getStoragePolicy();
            if (null != storagePolicy) {
                this.setSpaceLifecycle(bucketName, storagePolicy.getBucketLifecycleConfig());
            }
            return bucket;
        }
        catch (AmazonClientException e) {
            String err = "Could not create S3 bucket with name " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    private String getHiddenBucketName(String spaceId) {
        return HIDDEN_SPACE_PREFIX + this.getNewBucketName(spaceId);
    }

    public String createHiddenSpace(String spaceId, int expirationInDays) {
        String bucketName = this.getHiddenBucketName(spaceId);
        try {
            Bucket bucket = this.s3Client.createBucket(bucketName);
            BucketLifecycleConfiguration.Rule expiresRule = new BucketLifecycleConfiguration.Rule().withId("ExpirationRule").withExpirationInDays(expirationInDays).withStatus("Enabled");
            BucketLifecycleConfiguration configuration = new BucketLifecycleConfiguration().withRules(new BucketLifecycleConfiguration.Rule[]{expiresRule});
            this.s3Client.setBucketLifecycleConfiguration(bucketName, configuration);
            return spaceId;
        }
        catch (AmazonClientException e) {
            String err = "Could not create S3 bucket with name " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    protected StoragePolicy getStoragePolicy() {
        return new StoragePolicy(StorageClass.StandardInfrequentAccess, 30);
    }

    public void setSpaceLifecycle(String bucketName, BucketLifecycleConfiguration config) {
        boolean success = false;
        int maxLoops = 6;
        for (int loops = 0; !success && loops < maxLoops; ++loops) {
            try {
                this.s3Client.deleteBucketLifecycleConfiguration(bucketName);
                this.s3Client.setBucketLifecycleConfiguration(bucketName, config);
                success = true;
                continue;
            }
            catch (NotFoundException e) {
                success = false;
                this.wait(loops);
            }
        }
        if (!success) {
            throw new StorageException("Lifecycle policy for bucket " + bucketName + " could not be applied. The space cannot be found.");
        }
    }

    protected String getNewBucketName(String spaceId) {
        return S3ProviderUtil.createNewBucketName(this.accessKeyId, spaceId);
    }

    private String formattedDate(Date date) {
        return DateUtil.convertToString((long)date.getTime());
    }

    public void removeSpace(String spaceId) {
        String bucketName = this.getBucketName(spaceId);
        try {
            this.s3Client.deleteBucket(bucketName);
        }
        catch (AmazonClientException e) {
            String err = "Could not delete S3 bucket with name " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    protected Map<String, String> getAllSpaceProperties(String spaceId) {
        this.log.debug("getAllSpaceProperties(" + spaceId + ")");
        String bucketName = this.getBucketName(spaceId);
        Map<String, String> spaceProperties = new HashMap<String, String>();
        BucketTaggingConfiguration tagConfig = this.s3Client.getBucketTaggingConfiguration(bucketName);
        if (null != tagConfig) {
            for (TagSet tagSet : tagConfig.getAllTagSets()) {
                spaceProperties.putAll(tagSet.getAllTags());
            }
        }
        spaceProperties = this.replaceInMapValues(spaceProperties, "+", "@");
        spaceProperties.put("space-count", this.getSpaceCount(spaceId, 1000));
        return spaceProperties;
    }

    protected String getSpaceCount(String spaceId, int maxCount) {
        String marker;
        List<String> spaceContentChunk = null;
        long count = 0L;
        do {
            marker = null;
            if (spaceContentChunk == null || spaceContentChunk.size() <= 0) continue;
            marker = spaceContentChunk.get(spaceContentChunk.size() - 1);
        } while ((spaceContentChunk = this.getSpaceContentsChunked(spaceId, null, 1000L, marker)).size() > 0 && (count += (long)spaceContentChunk.size()) < (long)maxCount);
        String suffix = "";
        if (count >= (long)maxCount) {
            suffix = "+";
        }
        return String.valueOf(count) + suffix;
    }

    private String getBucketCreationDate(String bucketName) {
        Date created = null;
        try {
            List buckets = this.s3Client.listBuckets();
            for (Bucket bucket : buckets) {
                if (!bucket.getName().equals(bucketName)) continue;
                created = bucket.getCreationDate();
            }
        }
        catch (AmazonClientException e) {
            String err = "Could not retrieve S3 bucket listing due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
        String formattedDate = null;
        formattedDate = created != null ? this.formattedDate(created) : "unknown";
        return formattedDate;
    }

    protected void doSetSpaceProperties(String spaceId, Map<String, String> spaceProperties) {
        Map<String, String> originalProperties;
        this.log.debug("setSpaceProperties(" + spaceId + ")");
        String bucketName = this.getBucketName(spaceId);
        try {
            originalProperties = this.getAllSpaceProperties(spaceId);
        }
        catch (NotFoundException e) {
            originalProperties = new HashMap<String, String>();
        }
        String creationDate = originalProperties.get("space-created");
        if (creationDate == null && (creationDate = spaceProperties.get("space-created")) == null) {
            creationDate = this.getBucketCreationDate(bucketName);
        }
        spaceProperties.put("space-created", creationDate);
        spaceProperties = this.replaceInMapValues(spaceProperties, "@", "+");
        BucketTaggingConfiguration tagConfig = new BucketTaggingConfiguration().withTagSets(new TagSet[]{new TagSet(spaceProperties)});
        this.s3Client.setBucketTaggingConfiguration(bucketName, tagConfig);
    }

    private Map<String, String> replaceInMapValues(Map<String, String> map, String oldVal, String newVal) {
        for (String key : map.keySet()) {
            String value = map.get(key);
            if (!value.contains(oldVal)) continue;
            value = StringUtils.replace((String)value, (String)oldVal, (String)newVal);
            map.put(key, value);
        }
        return map;
    }

    public String addHiddenContent(String spaceId, String contentId, String contentMimeType, InputStream content) {
        this.log.debug("addHiddenContent(" + spaceId + ", " + contentId + ", " + contentMimeType + ")");
        String bucketName = this.getBucketName(spaceId);
        if (contentMimeType == null || contentMimeType.equals("")) {
            contentMimeType = "application/octet-stream";
        }
        ObjectMetadata objMetadata = new ObjectMetadata();
        objMetadata.setContentType(contentMimeType);
        PutObjectRequest putRequest = new PutObjectRequest(bucketName, contentId, content, objMetadata);
        putRequest.setStorageClass(DEFAULT_STORAGE_CLASS);
        putRequest.setCannedAcl(CannedAccessControlList.Private);
        try {
            PutObjectResult putResult = this.s3Client.putObject(putRequest);
            return putResult.getETag();
        }
        catch (AmazonClientException e) {
            String err = "Could not add content " + contentId + " with type " + contentMimeType + " to S3 bucket " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, false);
        }
    }

    public String addContent(String spaceId, String contentId, String contentMimeType, Map<String, String> userProperties, long contentSize, String contentChecksum, InputStream content) {
        String etag;
        ChecksumInputStream wrappedContent;
        block15: {
            this.log.debug("addContent(" + spaceId + ", " + contentId + ", " + contentMimeType + ", " + contentSize + ", " + contentChecksum + ")");
            String bucketName = this.getBucketName(spaceId);
            wrappedContent = new ChecksumInputStream(content, contentChecksum);
            String contentEncoding = this.removeContentEncoding(userProperties);
            userProperties = this.removeCalculatedProperties(userProperties);
            if (contentMimeType == null || contentMimeType.equals("")) {
                contentMimeType = "application/octet-stream";
            }
            ObjectMetadata objMetadata = new ObjectMetadata();
            objMetadata.setContentType(contentMimeType);
            if (contentSize > 0L) {
                objMetadata.setContentLength(contentSize);
            }
            if (null != contentChecksum && !contentChecksum.isEmpty()) {
                String encodedChecksum = ChecksumUtil.convertToBase64Encoding((String)contentChecksum);
                objMetadata.setContentMD5(encodedChecksum);
            }
            if (contentEncoding != null) {
                objMetadata.setContentEncoding(contentEncoding);
            }
            if (userProperties != null) {
                for (String key : userProperties.keySet()) {
                    String value = userProperties.get(key);
                    if (this.log.isDebugEnabled()) {
                        this.log.debug("[" + key + "|" + value + "]");
                    }
                    objMetadata.addUserMetadata(this.getSpaceFree(S3StorageProvider.encodeHeaderKey(key)), S3StorageProvider.encodeHeaderValue(value));
                }
            }
            PutObjectRequest putRequest = new PutObjectRequest(bucketName, contentId, (InputStream)wrappedContent, objMetadata);
            putRequest.setStorageClass(DEFAULT_STORAGE_CLASS);
            putRequest.setCannedAcl(CannedAccessControlList.Private);
            try {
                PutObjectResult putResult = this.s3Client.putObject(putRequest);
                etag = putResult.getETag();
            }
            catch (AmazonClientException e) {
                String err;
                if (e instanceof AmazonS3Exception) {
                    AmazonS3Exception s3Ex = (AmazonS3Exception)e;
                    String errorCode = s3Ex.getErrorCode();
                    Integer statusCode = s3Ex.getStatusCode();
                    String message = MessageFormat.format("exception putting object {0} into {1}: errorCode={2},  statusCode={3}, errorMessage={4}", contentId, bucketName, errorCode, statusCode, e.getMessage());
                    if (errorCode.equals("InvalidDigest") || errorCode.equals("BadDigest")) {
                        this.log.error(message, (Throwable)e);
                        String err2 = "Checksum mismatch detected attempting to add content " + contentId + " to S3 bucket " + bucketName + ". Content was not added.";
                        throw new ChecksumMismatchException(err2, (Throwable)e, false);
                    }
                    if (errorCode.equals("IncompleteBody")) {
                        this.log.error(message, (Throwable)e);
                        throw new StorageException("The content body was incomplete for " + contentId + " to S3 bucket " + bucketName + ". Content was not added.", (Throwable)e, false);
                    }
                    if (!statusCode.equals(503) && !statusCode.equals(404)) {
                        this.log.error(message, (Throwable)e);
                    } else {
                        this.log.warn(message, (Throwable)e);
                    }
                } else {
                    err = MessageFormat.format("exception putting object {0} into {1}: {2}", contentId, bucketName, e.getMessage());
                    this.log.error(err, (Throwable)e);
                }
                etag = this.doesContentExistWithExpectedChecksum(bucketName, contentId, contentChecksum);
                if (null != etag) break block15;
                err = "Could not add content " + contentId + " with type " + contentMimeType + " and size " + contentSize + " to S3 bucket " + bucketName + " due to error: " + e.getMessage();
                throw new StorageException(err, (Throwable)e, false);
            }
        }
        String providerChecksum = this.getETagValue(etag);
        String checksum = wrappedContent.getMD5();
        StorageProviderUtil.compareChecksum((String)providerChecksum, (String)spaceId, (String)contentId, (String)checksum);
        return providerChecksum;
    }

    private String removeContentEncoding(Map<String, String> properties) {
        if (properties != null) {
            return properties.remove("Content-Encoding");
        }
        return null;
    }

    protected String doesContentExistWithExpectedChecksum(String bucketName, String contentId, String expectedChecksum) {
        int maxAttempts = 20;
        int waitInSeconds = 2;
        int attempts = 0;
        int totalSecondsWaited = 0;
        String etag = null;
        for (int i = 0; i < maxAttempts; ++i) {
            try {
                ObjectMetadata metadata = this.s3Client.getObjectMetadata(bucketName, contentId);
                if (null != metadata) {
                    if (attempts > 5) {
                        this.log.info("contentId={} found in bucket={} after waiting for {} seconds...", new Object[]{contentId, bucketName, totalSecondsWaited});
                    }
                    if (expectedChecksum.equals(this.getETagValue(etag = metadata.getETag()))) {
                        return etag;
                    }
                }
            }
            catch (AmazonClientException metadata) {
                // empty catch block
            }
            ++attempts;
            int waitNow = waitInSeconds * i;
            this.wait(waitNow);
            totalSecondsWaited += waitNow;
        }
        if (etag == null) {
            this.log.warn("contentId={} NOT found in bucket={} after waiting for {} seconds...", new Object[]{contentId, bucketName, attempts * waitInSeconds});
        } else {
            this.log.warn("contentId={} in bucket={} does not have the expected checksum after waiting for {} seconds. S3 Checksum={} Expected Checksum={}", new Object[]{contentId, bucketName, attempts * waitInSeconds, this.getETagValue(etag), expectedChecksum});
        }
        return etag;
    }

    protected void wait(int seconds) {
        try {
            Thread.sleep(1000 * seconds);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public String copyContent(String sourceSpaceId, String sourceContentId, String destSpaceId, String destContentId) {
        this.log.debug("copyContent({}, {}, {}, {})", new Object[]{sourceSpaceId, sourceContentId, destSpaceId, destContentId});
        String sourceBucketName = this.getBucketName(sourceSpaceId);
        String destBucketName = this.getBucketName(destSpaceId);
        this.throwIfContentNotExist(sourceBucketName, sourceContentId);
        CopyObjectRequest request = new CopyObjectRequest(sourceBucketName, sourceContentId, destBucketName, destContentId);
        request.setStorageClass(DEFAULT_STORAGE_CLASS);
        request.setCannedAccessControlList(CannedAccessControlList.Private);
        CopyObjectResult result = this.doCopyObject(request);
        return StorageProviderUtil.compareChecksum((StorageProvider)this, (String)sourceSpaceId, (String)sourceContentId, (String)result.getETag());
    }

    private CopyObjectResult doCopyObject(CopyObjectRequest request) {
        try {
            return this.s3Client.copyObject(request);
        }
        catch (Exception e) {
            StringBuilder err = new StringBuilder("Error copying from: ");
            err.append(request.getSourceBucketName());
            err.append(" / ");
            err.append(request.getSourceKey());
            err.append(", to: ");
            err.append(request.getDestinationBucketName());
            err.append(" / ");
            err.append(request.getDestinationKey());
            this.log.error(err.toString() + "msg: {}", (Object)e.getMessage());
            throw new StorageException(err.toString(), (Throwable)e, true);
        }
    }

    public RetrievedContent getContent(String spaceId, String contentId) {
        return this.getContent(spaceId, contentId, null);
    }

    public RetrievedContent getContent(String spaceId, String contentId, String range) {
        this.log.debug("getContent(" + spaceId + ", " + contentId + ", " + range + ")");
        String bucketName = this.getBucketName(spaceId);
        try {
            GetObjectRequest getRequest = new GetObjectRequest(bucketName, contentId);
            if (StringUtils.isNotEmpty((String)range)) {
                ContentByteRange byteRange = new ContentByteRange(range);
                if (null == byteRange.getRangeStart()) {
                    throw new IllegalArgumentException(byteRange.getUsage(range));
                }
                if (null == byteRange.getRangeEnd()) {
                    getRequest.setRange(byteRange.getRangeStart().longValue());
                } else {
                    getRequest.setRange(byteRange.getRangeStart().longValue(), byteRange.getRangeEnd().longValue());
                }
            }
            S3Object contentItem = this.s3Client.getObject(getRequest);
            RetrievedContent retrievedContent = new RetrievedContent();
            retrievedContent.setContentStream((InputStream)contentItem.getObjectContent());
            retrievedContent.setContentProperties(this.prepContentProperties(contentItem.getObjectMetadata()));
            return retrievedContent;
        }
        catch (AmazonClientException e) {
            this.throwIfContentNotExist(bucketName, contentId);
            String err = "Could not retrieve content " + contentId + " in S3 bucket " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    public void deleteContent(String spaceId, String contentId) {
        this.log.debug("deleteContent(" + spaceId + ", " + contentId + ")");
        String bucketName = this.getBucketName(spaceId);
        this.throwIfContentNotExist(bucketName, contentId);
        try {
            this.s3Client.deleteObject(bucketName, contentId);
        }
        catch (AmazonClientException e) {
            String err = "Could not delete content " + contentId + " from S3 bucket " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, true);
        }
    }

    public void setContentProperties(String spaceId, String contentId, Map<String, String> contentProperties) {
        Map<String, String> existingMeta;
        String existingMime;
        this.log.debug("setContentProperties(" + spaceId + ", " + contentId + ")");
        String bucketName = this.getBucketName(spaceId);
        String contentEncoding = this.removeContentEncoding(contentProperties);
        contentProperties = this.removeCalculatedProperties(contentProperties);
        String mimeType = contentProperties.remove("content-mimetype");
        if ((mimeType == null || mimeType.equals("")) && (existingMime = (existingMeta = this.getContentProperties(spaceId, contentId)).get("content-mimetype")) != null) {
            mimeType = existingMime;
        }
        ObjectMetadata objMetadata = new ObjectMetadata();
        for (String key : contentProperties.keySet()) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("[" + key + "|" + contentProperties.get(key) + "]");
            }
            objMetadata.addUserMetadata(this.getSpaceFree(key), contentProperties.get(key));
        }
        if (mimeType != null && !mimeType.equals("")) {
            objMetadata.setContentType(mimeType);
        }
        if (contentEncoding != null && !contentEncoding.equals("")) {
            objMetadata.setContentEncoding(contentEncoding);
        }
        this.updateObjectProperties(bucketName, contentId, objMetadata);
    }

    protected Map<String, String> removeCalculatedProperties(Map<String, String> contentProperties) {
        if ((contentProperties = super.removeCalculatedProperties(contentProperties)) != null) {
            contentProperties.remove("Content-Length");
            contentProperties.remove("Content-Type");
            contentProperties.remove("Last-Modified");
            contentProperties.remove("Date");
            contentProperties.remove("ETag");
            contentProperties.remove("Content-Length".toLowerCase());
            contentProperties.remove("Content-Type".toLowerCase());
            contentProperties.remove("Last-Modified".toLowerCase());
            contentProperties.remove("Date".toLowerCase());
            contentProperties.remove("ETag".toLowerCase());
        }
        return contentProperties;
    }

    private void throwIfContentNotExist(String bucketName, String contentId) {
        try {
            this.s3Client.getObjectMetadata(bucketName, contentId);
        }
        catch (AmazonClientException e) {
            String err = "Could not find content item with ID " + contentId + " in S3 bucket " + bucketName + ". S3 error: " + e.getMessage();
            throw new NotFoundException(err);
        }
    }

    private ObjectMetadata getObjectDetails(String bucketName, String contentId, boolean retry) {
        try {
            return this.s3Client.getObjectMetadata(bucketName, contentId);
        }
        catch (AmazonClientException e) {
            this.throwIfContentNotExist(bucketName, contentId);
            String err = "Could not get details for content " + contentId + " in S3 bucket " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, retry);
        }
    }

    private void updateObjectProperties(String bucketName, String contentId, ObjectMetadata objMetadata) {
        try {
            AccessControlList originalACL = this.s3Client.getObjectAcl(bucketName, contentId);
            CopyObjectRequest copyRequest = new CopyObjectRequest(bucketName, contentId, bucketName, contentId);
            copyRequest.setStorageClass(DEFAULT_STORAGE_CLASS);
            copyRequest.setNewObjectMetadata(objMetadata);
            this.s3Client.copyObject(copyRequest);
            this.s3Client.setObjectAcl(bucketName, contentId, originalACL);
        }
        catch (AmazonClientException e) {
            this.throwIfContentNotExist(bucketName, contentId);
            String err = "Could not update metadata for content " + contentId + " in S3 bucket " + bucketName + " due to error: " + e.getMessage();
            throw new StorageException(err, (Throwable)e, false);
        }
    }

    public Map<String, String> getContentProperties(String spaceId, String contentId) {
        this.log.debug("getContentProperties(" + spaceId + ", " + contentId + ")");
        String bucketName = this.getBucketName(spaceId);
        ObjectMetadata objMetadata = this.getObjectDetails(bucketName, contentId, true);
        if (objMetadata == null) {
            String err = "No metadata is available for item " + contentId + " in S3 bucket " + bucketName;
            throw new StorageException(err, false);
        }
        return this.prepContentProperties(objMetadata);
    }

    public Map<String, String> getSpaceProperties(String spaceId) {
        return super.getSpaceProperties(spaceId);
    }

    private Map<String, String> prepContentProperties(ObjectMetadata objMetadata) {
        Date modified;
        String checksum;
        long contentLength;
        String encoding;
        HashMap<String, String> contentProperties = new HashMap<String, String>();
        Map userProperties = objMetadata.getUserMetadata();
        for (Object metaName : userProperties.keySet()) {
            String metaValue = (String)userProperties.get(metaName);
            contentProperties.put(this.getWithSpace(S3StorageProvider.decodeHeaderKey((String)metaName)), S3StorageProvider.decodeHeaderValue(metaValue));
        }
        Map responseMeta = objMetadata.getRawMetadata();
        for (String metaName : responseMeta.keySet()) {
            Object metaValue = responseMeta.get(metaName);
            if (!(metaValue instanceof String)) continue;
            contentProperties.put(metaName, (String)metaValue);
        }
        String contentType = objMetadata.getContentType();
        if (contentType != null) {
            contentProperties.put("content-mimetype", contentType);
            contentProperties.put("Content-Type", contentType);
        }
        if ((encoding = objMetadata.getContentEncoding()) != null) {
            contentProperties.put("Content-Encoding", encoding);
        }
        if ((contentLength = objMetadata.getContentLength()) >= 0L) {
            String size = String.valueOf(contentLength);
            contentProperties.put("content-size", size);
            contentProperties.put("Content-Length", size);
        }
        if ((checksum = objMetadata.getETag()) != null) {
            String eTagValue = this.getETagValue(checksum);
            contentProperties.put("content-checksum", eTagValue);
            contentProperties.put("content-md5", eTagValue);
            contentProperties.put("ETag", eTagValue);
        }
        if ((modified = objMetadata.getLastModified()) != null) {
            String modDate = this.formattedDate(modified);
            contentProperties.put("content-modified", modDate);
            contentProperties.put("Last-Modified", modDate);
        }
        return contentProperties;
    }

    protected String getETagValue(String etag) {
        String checksum = etag;
        if (checksum != null && checksum.indexOf("\"") == 0 && checksum.lastIndexOf("\"") == checksum.length() - 1) {
            checksum = checksum.substring(1, checksum.length() - 1);
        }
        return checksum;
    }

    public String getBucketName(String spaceId) {
        List<Bucket> buckets = this.listAllBuckets();
        for (Bucket bucket : buckets) {
            String bucketName = bucket.getName();
            spaceId = spaceId.replace(".", "[.]");
            if (!bucketName.matches("(hidden-)?[\\w]{20}[.]" + spaceId)) continue;
            return bucketName;
        }
        throw new NotFoundException("No S3 bucket found matching spaceID: " + spaceId);
    }

    protected String getSpaceId(String bucketName) {
        String spaceId = bucketName;
        if (this.isSpace(bucketName)) {
            spaceId = spaceId.substring(this.accessKeyId.length() + 1);
        }
        return spaceId;
    }

    protected boolean isSpace(String bucketName) {
        boolean isSpace = false;
        if (bucketName.matches("[\\w]{20}[.].*")) {
            isSpace = true;
        }
        return isSpace;
    }

    protected String getSpaceFree(String name) {
        return name.replaceAll(" ", "%20");
    }

    protected String getWithSpace(String name) {
        return name.replaceAll("%20", " ");
    }

    protected static String encodeHeaderValue(String userMetaValue) {
        try {
            String encodedValue = HEADER_VALUE_PREFIX + URLEncoder.encode(userMetaValue, UTF_8);
            return encodedValue;
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    protected static String decodeHeaderValue(String userMetaValue) {
        if (userMetaValue.startsWith(HEADER_VALUE_PREFIX)) {
            try {
                String encodedValue = URLDecoder.decode(userMetaValue.substring(HEADER_VALUE_PREFIX.length()), UTF_8);
                return encodedValue;
            }
            catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }
        }
        return userMetaValue;
    }

    protected static String encodeHeaderKey(String userMetaName) {
        return userMetaName + HEADER_KEY_SUFFIX;
    }

    protected static String decodeHeaderKey(String userMetaName) {
        if (userMetaName.endsWith(HEADER_KEY_SUFFIX)) {
            return userMetaName.substring(0, userMetaName.length() - 1);
        }
        return userMetaName;
    }
}

