/*
 * Decompiled with CFR 0.152.
 */
package org.duraspace.bagit;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import gov.loc.repository.bagit.domain.Bag;
import gov.loc.repository.bagit.domain.Manifest;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.duraspace.bagit.BagConfig;
import org.duraspace.bagit.ProfileFieldRule;
import org.duraspace.bagit.ProfileValidationException;
import org.duraspace.bagit.ProfileValidationUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BagProfile {
    private static final Logger logger = LoggerFactory.getLogger(BagProfile.class);
    private boolean allowFetch;
    private Serialization serialization;
    private Set<String> acceptedBagItVersions;
    private Set<String> acceptedSerializations;
    private Set<String> tagFilesAllowed;
    private Set<String> tagFilesRequired;
    private Set<String> allowedPayloadAlgorithms;
    private Set<String> allowedTagAlgorithms;
    private Set<String> payloadDigestAlgorithms;
    private Set<String> tagDigestAlgorithms;
    private Set<String> sections = new HashSet<String>();
    private Map<String, Map<String, ProfileFieldRule>> metadataFields = new HashMap<String, Map<String, ProfileFieldRule>>();
    private Map<String, String> profileMetadata = new HashMap<String, String>();

    public BagProfile(BuiltIn builtInProfile) throws IOException {
        String resource = "profiles/" + builtInProfile.identifier + ".json";
        URL resourceURL = this.getClass().getClassLoader().getResource(resource);
        try (InputStream in = Objects.requireNonNull(resourceURL).openStream();){
            this.load(in);
        }
    }

    public BagProfile(InputStream in) throws IOException {
        this.load(in);
    }

    private void load(InputStream in) throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        JsonNode json = mapper.readTree(in);
        this.loadProfileInfo(json);
        this.allowFetch = json.has("Allow-Fetch.txt") ? json.get("Allow-Fetch.txt").asBoolean() : true;
        this.serialization = json.has("Serialization") ? Serialization.of(json.get("Serialization").asText()) : Serialization.OPTIONAL;
        this.acceptedBagItVersions = BagProfile.arrayValues(json, "Accept-BagIt-Version");
        this.acceptedSerializations = BagProfile.arrayValues(json, "Accept-Serialization");
        this.tagFilesAllowed = BagProfile.arrayValues(json, "Tag-Files-Allowed");
        this.tagFilesRequired = BagProfile.arrayValues(json, "Tag-Files-Required");
        this.allowedPayloadAlgorithms = BagProfile.arrayValues(json, "Manifests-Allowed");
        this.allowedTagAlgorithms = BagProfile.arrayValues(json, "Tag-Manifests-Allowed");
        this.payloadDigestAlgorithms = BagProfile.arrayValues(json, "Manifests-Required");
        this.tagDigestAlgorithms = BagProfile.arrayValues(json, "Tag-Manifests-Required");
        this.metadataFields.put("Bag-Info", BagProfile.metadataFields(json, "Bag-Info"));
        this.sections.add("Bag-Info");
        if (json.get("Other-Info") != null) {
            this.loadOtherTags(json);
        }
    }

    private void loadProfileInfo(JsonNode json) {
        JsonNode tag = json.get("BagIt-Profile-Info");
        if (tag != null) {
            tag.fields().forEachRemaining(entry -> this.profileMetadata.put((String)entry.getKey(), ((JsonNode)entry.getValue()).asText()));
        }
    }

    private void loadOtherTags(JsonNode json) {
        JsonNode arrayTags = json.get("Other-Info");
        if (arrayTags != null && arrayTags.isArray()) {
            arrayTags.forEach(tag -> tag.fieldNames().forEachRemaining(this.sections::add));
            Iterator arrayEntries = arrayTags.elements();
            while (arrayEntries.hasNext()) {
                JsonNode entries = (JsonNode)arrayEntries.next();
                Iterator tagNames = entries.fieldNames();
                while (tagNames.hasNext()) {
                    String tagName = (String)tagNames.next();
                    this.metadataFields.put(tagName, BagProfile.metadataFields(entries, tagName));
                }
            }
        }
        logger.debug("tagFiles is {}", this.sections);
        logger.debug("metadataFields is {}", this.metadataFields);
    }

    private static Set<String> arrayValues(JsonNode json, String key) {
        JsonNode values = json.get(key);
        if (values == null) {
            return Collections.emptySet();
        }
        HashSet<String> results = new HashSet<String>();
        for (int i = 0; i < values.size(); ++i) {
            results.add(values.get(i).asText());
        }
        return results;
    }

    private static Map<String, ProfileFieldRule> metadataFields(JsonNode json, String key) {
        JsonNode fields = json.get(key);
        if (fields == null) {
            return Collections.emptyMap();
        }
        HashMap<String, ProfileFieldRule> results = new HashMap<String, ProfileFieldRule>();
        Iterator it = fields.fieldNames();
        while (it.hasNext()) {
            JsonNode descriptionNode;
            JsonNode recommendedNode;
            JsonNode repeatedNode;
            boolean required = false;
            boolean repeatable = true;
            boolean recommended = false;
            String description = "No description";
            String name = (String)it.next();
            JsonNode field = fields.get(name);
            JsonNode requiredNode = field.get("required");
            if (requiredNode != null && requiredNode.asBoolean()) {
                required = requiredNode.asBoolean();
            }
            if ((repeatedNode = field.get("repeatable")) != null) {
                repeatable = repeatedNode.asBoolean();
            }
            if ((recommendedNode = field.get("recommended")) != null && recommendedNode.asBoolean()) {
                recommended = recommendedNode.asBoolean();
            }
            if ((descriptionNode = field.get("description")) != null && descriptionNode.asText().isEmpty()) {
                description = descriptionNode.asText();
            }
            Set<String> values = BagProfile.arrayValues(field, "values");
            results.put(name, new ProfileFieldRule(required, repeatable, recommended, description, values));
        }
        return results;
    }

    public boolean isAllowFetch() {
        return this.allowFetch;
    }

    public Serialization getSerialization() {
        return this.serialization;
    }

    public Set<String> getAcceptedBagItVersions() {
        return this.acceptedBagItVersions;
    }

    public Set<String> getAcceptedSerializations() {
        return this.acceptedSerializations;
    }

    public Set<String> getTagFilesAllowed() {
        return this.tagFilesAllowed;
    }

    public Set<String> getTagFilesRequired() {
        return this.tagFilesRequired;
    }

    public Set<String> getAllowedPayloadAlgorithms() {
        return this.allowedPayloadAlgorithms;
    }

    public Set<String> getAllowedTagAlgorithms() {
        return this.allowedTagAlgorithms;
    }

    public Set<String> getPayloadDigestAlgorithms() {
        return this.payloadDigestAlgorithms;
    }

    public Set<String> getTagDigestAlgorithms() {
        return this.tagDigestAlgorithms;
    }

    public Map<String, ProfileFieldRule> getMetadataFields() {
        return this.getMetadataFields("Bag-Info");
    }

    public Map<String, ProfileFieldRule> getMetadataFields(String tagFile) {
        return this.metadataFields.get(tagFile);
    }

    public Set<String> getSectionNames() {
        return this.sections;
    }

    public Map<String, String> getProfileMetadata() {
        return this.profileMetadata;
    }

    public void validateConfig(BagConfig config) {
        for (String section : this.sections) {
            String tagFile = section.toLowerCase() + ".txt";
            if (config.hasTagFile(tagFile)) {
                try {
                    ProfileValidationUtil.validate(section, this.getMetadataFields(section), config.getFieldsForTagFile(tagFile));
                    ProfileValidationUtil.validateTagIsAllowed(Paths.get(tagFile, new String[0]), this.tagFilesAllowed);
                    continue;
                }
                catch (ProfileValidationException e) {
                    throw new RuntimeException(e.getMessage(), e);
                }
            }
            throw new RuntimeException(String.format("Error missing section %s from bag config", section));
        }
    }

    public void validateBag(Bag bag) {
        logger.info("Starting Bag to BagProfile conformance validator");
        String tagIdentifier = "tag";
        String fetchIdentifier = "fetch.txt";
        String payloadIdentifier = "payload";
        StringBuilder errors = new StringBuilder();
        Path root = bag.getRootDir();
        Set foundPayloadManifests = bag.getPayLoadManifests();
        Set foundTagManifests = bag.getTagManifests();
        if (!(this.allowFetch || bag.getItemsToFetch().isEmpty() && !Files.exists(root.resolve("fetch.txt"), new LinkOption[0]))) {
            errors.append("Profile does not allow a fetch.txt but fetch file found!\n");
        }
        errors.append(ProfileValidationUtil.validateManifest(foundPayloadManifests, this.payloadDigestAlgorithms, this.allowedPayloadAlgorithms, "payload"));
        if (foundTagManifests.isEmpty()) {
            errors.append("No tag manifest found!\n");
        } else {
            errors.append(ProfileValidationUtil.validateManifest(foundTagManifests, this.tagDigestAlgorithms, this.allowedTagAlgorithms, "tag"));
            Manifest manifest = (Manifest)foundTagManifests.iterator().next();
            Set existingTagFiles = manifest.getFileToChecksumMap().keySet();
            for (Path tag : existingTagFiles) {
                Path relativePath = tag.startsWith(root) ? root.relativize(tag) : tag;
                try {
                    ProfileValidationUtil.validateTagIsAllowed(relativePath, this.tagFilesAllowed);
                }
                catch (ProfileValidationException e) {
                    errors.append(e.getMessage());
                }
            }
        }
        for (String tagName : this.tagFilesRequired) {
            Path requiredTag = root.resolve(tagName);
            if (requiredTag.toFile().exists()) continue;
            errors.append("Required tag file \"").append(tagName).append("\" does not exist!\n");
        }
        for (String section : this.sections) {
            String tagFile = section.toLowerCase() + ".txt";
            Path resolved = root.resolve(tagFile);
            try {
                ProfileValidationUtil.validate(section, this.metadataFields.get(section), resolved);
            }
            catch (IOException e) {
                errors.append("Could not read info from \"").append(tagFile).append("\"!\n");
            }
            catch (ProfileValidationException e) {
                errors.append(e.getMessage());
            }
        }
        if (!this.acceptedBagItVersions.contains(bag.getVersion().toString())) {
            errors.append("BagIt version incompatible; accepted versions are ").append(this.acceptedBagItVersions).append("\n");
        }
        if (this.serialization == Serialization.REQUIRED) {
            logger.warn("Bag Profile requires serialization, import will continue if the bag has been deserialized");
        }
        if (errors.length() > 0) {
            throw new RuntimeException("Bag profile validation failure: The following errors occurred: \n" + errors.toString());
        }
    }

    public static enum BuiltIn {
        APTRUST("aptrust"),
        BEYOND_THE_REPOSITORY("beyondtherepository"),
        DEFAULT("default"),
        METAARCHIVE("metaarchive"),
        PERSEIDS("perseids");

        private final String identifier;

        private BuiltIn(String identifier) {
            this.identifier = identifier;
        }

        public static BuiltIn from(String identifier) {
            switch (identifier.toLowerCase()) {
                case "aptrust": {
                    return APTRUST;
                }
                case "beyondtherepository": {
                    return BEYOND_THE_REPOSITORY;
                }
                case "default": {
                    return DEFAULT;
                }
                case "metaarchive": {
                    return METAARCHIVE;
                }
                case "perseids": {
                    return PERSEIDS;
                }
            }
            throw new IllegalArgumentException("Unsupported profile identifier. Accepted values are: " + Arrays.stream(BuiltIn.values()).map(BuiltIn::getIdentifier).collect(Collectors.joining(", ")));
        }

        public String getIdentifier() {
            return this.identifier;
        }
    }

    public static enum Serialization {
        FORBIDDEN,
        REQUIRED,
        OPTIONAL,
        UNKNOWN;


        public static Serialization of(String value) {
            switch (value.toLowerCase()) {
                case "forbidden": {
                    return FORBIDDEN;
                }
                case "required": {
                    return REQUIRED;
                }
                case "optional": {
                    return OPTIONAL;
                }
            }
            return UNKNOWN;
        }
    }
}

