package org.duraspace.fcrepo.cloudsync.service.dao;

import org.duraspace.fcrepo.cloudsync.api.ObjectSet;
import org.duraspace.fcrepo.cloudsync.service.backend.ObjectQuery;
import org.duraspace.fcrepo.cloudsync.service.util.StringUtil;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

public class ObjectSetDao extends AbstractDao {

    private static final String CREATE_DDL =
            "CREATE TABLE ObjectSets (\n"
            + "  id INTEGER PRIMARY KEY NOT NULL GENERATED BY DEFAULT AS IDENTITY,\n"
            + "  name VARCHAR(256) NOT NULL,\n"
            + "  type VARCHAR(32) NOT NULL,\n"
            + "  data VARCHAR(32672) NOT NULL,\n"
            + "  CONSTRAINT ObjectSetUniqueName unique (name))";

    private static final String UPDATE_SQL =
            "UPDATE ObjectSets\n"
            + "SET name = ?, type= ?, data = ?\n"
            + "WHERE id = ?";

    public ObjectSetDao(JdbcTemplate db) {
        super(db);
    }

    @Override
    public void initDb() {
        db.execute(CREATE_DDL);
        ObjectSet o = new ObjectSet();
        o.setName("All Objects");
        o.setType("pidPattern");
        o.setData("*");
        createObjectSet(o);
    }

    public ObjectSet createObjectSet(ObjectSet objectSet)
            throws DuplicateKeyException {
        // normalize and validate fields
        if (StringUtil.normalize(objectSet.getId()) != null) {
            throw new IllegalArgumentException("Specifying the Object Set "
                    + "id during creation is not permitted");
        }
        objectSet.setName(StringUtil.validate("name", objectSet.getName(), 256));
        objectSet.setType(StringUtil.validate("type", objectSet.getType(), 32));
        objectSet.setData(StringUtil.validate("data", objectSet.getData(), 32672));
        new ObjectQuery(objectSet); // do type-specific validation
        String id = insert(
                "INSERT INTO ObjectSets (name, type, data) VALUES (?, ?, ?)",
                objectSet.getName(),
                objectSet.getType(),
                objectSet.getData());
        return getObjectSet(id);
    }

    public List<ObjectSet> listObjectSets() {
        return db.query("SELECT * FROM ObjectSets",
                new RowMapper<ObjectSet>() {
                    public ObjectSet mapRow(ResultSet rs, int i) throws SQLException {
                        return getObjectSet(rs);
                    }
                });
    }

    public ObjectSet getObjectSet(String id) {
        return db.query("SELECT * FROM ObjectSets WHERE id = ?",
                new ResultSetExtractor<ObjectSet>() {
                    public ObjectSet extractData(ResultSet rs)
                            throws SQLException {
                        if (rs.next()) {
                            return getObjectSet(rs);
                        } else {
                            return null;
                        }
                    }
                },
                Integer.parseInt(id));
    }

    private static ObjectSet getObjectSet(ResultSet rs) throws SQLException {
        ObjectSet o = new ObjectSet();
        o.setId("" + rs.getInt("id"));
        o.setName(rs.getString("name"));
        o.setType(rs.getString("type"));
        o.setData(rs.getString("data"));
        return o;
    }

    public ObjectSet updateObjectSet(String id, ObjectSet set)
            throws DuplicateKeyException {
        // Validate the changes implied by the given Task object,
        // fully populating a newTask object along the way.
        ObjectSet orig = getObjectSet(id);
        // id and activeLogId are never user-settable
        String newId = StringUtil.normalize(set.getId());
        if (newId != null && !newId.equals(id)) {
            throw new UnsupportedOperationException("Changing the ObjectSet id is not permitted");
        }
        updateObjectSet(orig, set);

        return getObjectSet(orig.getId());
    }

    private void updateObjectSet(final ObjectSet orig, ObjectSet mods) {
        if (StringUtil.normalize(mods.getName()) != null) {
            orig.setName(StringUtil.validate("name", mods.getName(), 256));
        }
        if (StringUtil.normalize(mods.getType()) != null) {
            orig.setType(StringUtil.validate("type", mods.getType(), 32));
        }

        if (StringUtil.normalize(mods.getData()) != null) {
            orig.setData(StringUtil.validate("data", mods.getData(), 32672));
        }

        // Finally, update the necessary table
        final int objectSetId = Integer.parseInt(orig.getId());
        new ObjectQuery(orig);
        db.update(UPDATE_SQL,
                orig.getName(),
                orig.getType(),
                orig.getData(),
                objectSetId);
    }

    public void deleteObjectSet(String id) {
        db.update("DELETE FROM ObjectSets WHERE id = ?", Integer.parseInt(id));
    }
}
