package org.duraspace.fcrepo.cloudsync.service.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.springframework.dao.DuplicateKeyException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionTemplate;

import org.duraspace.fcrepo.cloudsync.api.Task;
import org.duraspace.fcrepo.cloudsync.service.backend.TaskRunner;
import org.duraspace.fcrepo.cloudsync.service.util.StringUtil;
import com.github.cwilper.fcrepo.httpclient.HttpClientConfig;

public class TaskDao extends AbstractDao {

    private static final String CREATE_DDL =
          "CREATE TABLE Tasks (\n"
        + "  id INTEGER PRIMARY KEY NOT NULL GENERATED BY DEFAULT AS IDENTITY,\n"
        + "  name VARCHAR(256) NOT NULL,\n"
        + "  type VARCHAR(32) NOT NULL,\n"
        + "  state VARCHAR(16) NOT NULL,\n"
        + "  activeLogId INTEGER,\n"
        + "  schedule VARCHAR(1024),\n"
        + "  data VARCHAR(32672) NOT NULL,\n"
        + "  CONSTRAINT TaskUniqueName UNIQUE (name))";
        // + "  CONSTRAINT Task_Log_FK FOREIGN_KEY (activeLogId) REFERENCES TaskLogs (id))";

    private static final String CREATE_DDL2 =
          "CREATE TABLE TaskSetDeps (\n"
        + "  taskId INTEGER NOT NULL,\n"
        + "  setId INTEGER NOT NULL,\n"
        + "  CONSTRAINT TaskSetDep_Task_FK FOREIGN KEY (taskId) REFERENCES Tasks (id),\n"
        + "  CONSTRAINT TaskSetDep_Set_FK FOREIGN KEY (setId) REFERENCES ObjectSets (id))";

    private static final String CREATE_DDL3 =
          "CREATE TABLE TaskStoreDeps (\n"
        + "  taskId INTEGER NOT NULL,\n"
        + "  storeId INTEGER NOT NULL,\n"
        + "  CONSTRAINT TaskStoreDep_Task_FK FOREIGN KEY (taskId) REFERENCES Tasks (id),\n"
        + "  CONSTRAINT TaskStoreDep_Store_FK FOREIGN KEY (storeId) REFERENCES ObjectStores (id))";

    private static final String INSERT_SQL =
          "INSERT INTO Tasks (name, type, state, schedule, data) \n"
        + "VALUES (?, ?, ?, ?, ?)";

    private static final String INSERT_SQL2 =
          "INSERT INTO TaskSetDeps (taskId, setId) VALUES (?, ?)";

    private static final String INSERT_SQL3 =
          "INSERT INTO TaskStoreDeps (taskId, storeId) VALUES (?, ?)";

    private static final String UPDATE_SQL =
          "UPDATE Tasks\n"
        + "SET name = ?, type = ?, state = ?, schedule = ?, data = ?\n"
        + "WHERE id = ?";

    private static final String UPDATE_SQL2 =
          "UPDATE Tasks SET state = ? WHERE id = ?";

    private final TransactionTemplate tt;

    private final ObjectSetDao objectSetDao;

    private final ObjectStoreDao objectStoreDao;

    public TaskDao(JdbcTemplate db, TransactionTemplate tt,
                   ObjectSetDao objectSetDao,
                   ObjectStoreDao objectStoreDao) {
        super(db);
        this.tt = tt;
        this.objectSetDao = objectSetDao;
        this.objectStoreDao = objectStoreDao;
    }

    @Override
    public void initDb() {
        db.execute(CREATE_DDL);
        db.execute(CREATE_DDL2);
        db.execute(CREATE_DDL3);
    }

    public Task createTask(final Task task) {
        // normalize and validate fields
        if (StringUtil.normalize(task.getId()) != null) {
            throw new IllegalArgumentException("Specifying the Task "
                    + "id during creation is not permitted");
        }
        task.setName(StringUtil.validate("name", task.getName(), 256));
        task.setType(StringUtil.validate("type", task.getType(), 32));
        task.setState(StringUtil.validate("state", task.getState(),
                new String[] { Task.IDLE, Task.STARTING }));
        if (StringUtil.normalize(task.getActiveLogId()) != null) {
            throw new IllegalArgumentException("Specifying the Task "
                    + "activeLogId during creation is not permitted");
        }
        String sched = StringUtil.normalize(task.getSchedule());
        if (sched != null) {
            task.setSchedule(StringUtil.validate("schedule", sched, 1024));
        }
        task.setData(StringUtil.validate("data", task.getData(), 32672));
        // Validate schedule content, type-specific data,
        // and do dependency determination
        final TaskRunner runner = TaskRunner.getInstance(task, this,
                objectSetDao, objectStoreDao, null, null, new HttpClientConfig());

        // persist task and deps, validating that the deps (foreign keys) exist
        String id = tt.execute(new TransactionCallback<String>() {
            public String doInTransaction(TransactionStatus status) {
                boolean success = false;
                try {
                    String id = insert(INSERT_SQL,
                            task.getName(),
                            task.getType(),
                            task.getState(),
                            task.getSchedule(),
                            task.getData());
                    Integer taskId = Integer.parseInt(id);
                    for (String setId: runner.getRelatedSetIds()) {
                        db.update(INSERT_SQL2, taskId,
                                Integer.parseInt(setId));
                    }
                    for (String storeId: runner.getRelatedStoreIds()) {
                        db.update(INSERT_SQL3, taskId, 
                                Integer.parseInt(storeId));
                    }
                    success = true;
                    return id;
                } finally {
                    if (!success) {
                        status.setRollbackOnly();
                    }
                }
            }
        });
        return getTask(id);
    }

    public List<Task> listTasks() {
        return db.query("SELECT * FROM Tasks",
                new RowMapper<Task>() {
                    public Task mapRow(ResultSet rs, int i) throws SQLException {
                        return getTask(rs);
                    }
                });
    }

    public Task getTask(String id) {
        return db.query("SELECT * FROM Tasks WHERE id = ?",
                new ResultSetExtractor<Task>() {
                    public Task extractData(ResultSet rs) throws SQLException {
                        if (rs.next()) {
                            return getTask(rs);
                        } else {
                            return null;
                        }
                    }
                },
                Integer.parseInt(id));
    }

    private static Task getTask(ResultSet rs) throws SQLException {
        Task task = new Task();
        task.setId("" + rs.getInt("id"));
        task.setName(rs.getString("name"));
        task.setType(rs.getString("type"));
        task.setState(rs.getString("state"));
        Integer activeLogId = rs.getInt("activeLogId");
        if (activeLogId != null) {
            task.setActiveLogId(activeLogId.toString());
        }
        task.setSchedule(rs.getString("schedule"));
        task.setData(rs.getString("data"));
        return task;
    }

    // TaskManager use only
    public void setTaskState(String id, String newState) {
        db.update(UPDATE_SQL2, newState, Integer.parseInt(id));
    }

    // TaskManager use only
    public void goIdle(String id) {
        setTaskState(id, Task.IDLE);
        setActiveLogId(id, null);
    }

    // TaskManager use only
    public void setActiveLogId(String id, String activeLogId) {
        Integer value = null;
        if (activeLogId != null) {
            value = Integer.parseInt(activeLogId);
        }
        db.update("UPDATE Tasks SET activeLogId = ? WHERE id = ?",
                value, Integer.parseInt(id));
    }

    public Task updateTask(String id, Task task)
            throws DuplicateKeyException {
        // Validate the changes implied by the given Task object,
        // fully populating a newTask object along the way.
        Task orig = getTask(id);
        // id and activeLogId are never user-settable
        String newId = StringUtil.normalize(task.getId());
        if (newId != null && !newId.equals(id)) {
            throw new UnsupportedOperationException("Changing the task id is not permitted");
        }
        String newActiveLogId = StringUtil.normalize(task.getActiveLogId());
        if (newActiveLogId != null) {
            String oldActiveLogId = orig.getActiveLogId();
            if (oldActiveLogId == null || !oldActiveLogId.equals(newActiveLogId)) {
                throw new UnsupportedOperationException("Changing the task activeLogId is not permitted");
            }
        }

        if (orig.getState().equals(Task.IDLE)) {
            updateIdleTask(orig, task);
        } else {
            updateActiveTask(orig, task);
        }
        return getTask(orig.getId());
    }

    // allow prop changes and state transition to "starting"
    private void updateIdleTask(final Task orig, Task mods) {
        // only allow transition to "starting" state
        String newState = StringUtil.normalize(mods.getState());
        if (newState != null && !newState.equals(Task.IDLE) && !newState.equals(Task.STARTING)) {
            throw new IllegalArgumentException("Illegal state transition: " + Task.IDLE + " -> " + newState);
        }
        if (newState != null) {
            orig.setState(newState);
        }

        // At this point, id, activeLogId, and state have been validated/set.
        // Now we need to validate/set name, type, schedule, and data.
        if (StringUtil.normalize(mods.getName()) != null) {
            orig.setName(StringUtil.validate("name", mods.getName(), 256));
        }
        if (StringUtil.normalize(mods.getType()) != null) {
            orig.setType(StringUtil.validate("type", mods.getType(), 32));
        }
        if (StringUtil.normalize(mods.getSchedule()) != null) {
            orig.setSchedule(StringUtil.validate("schedule", mods.getType(), 1024));
        }
        if (StringUtil.normalize(mods.getData()) != null) {
            orig.setData(StringUtil.validate("data", mods.getData(), 32672));
        }

        // Validate schedule content, type-specific data,
        // and do dependency determination
        final TaskRunner runner = TaskRunner.getInstance(orig, this,
                objectSetDao, objectStoreDao, null, null, new HttpClientConfig());

        // Finally, update the necessary tables in a transaction
        final int taskId = Integer.parseInt(orig.getId());
        tt.execute(new TransactionCallbackWithoutResult() {
            public void doInTransactionWithoutResult(TransactionStatus status) {
                boolean success = false;
                try {
                    db.update(UPDATE_SQL,
                            orig.getName(),
                            orig.getType(),
                            orig.getState(),
                            orig.getSchedule(),
                            orig.getData(),
                            taskId);
                    db.update("DELETE FROM TaskSetDeps WHERE taskId = ?", taskId);
                    db.update("DELETE FROM TaskStoreDeps WHERE taskId = ?", taskId);
                    for (String setId: runner.getRelatedSetIds()) {
                        db.update(INSERT_SQL2, taskId, Integer.parseInt(setId));
                    }
                    for (String storeId: runner.getRelatedStoreIds()) {
                        db.update(INSERT_SQL3, taskId, Integer.parseInt(storeId));
                    }
                    success = true;
                } finally {
                    if (!success) {
                        status.setRollbackOnly();
                    }
                }
            }
        });
    }

    // Disallow prop changes. Only allow the following state transitions:
    //
    // (STARTING, RUNNING, RESUMING) -> (PAUSING, CANCELING)
    // (PAUSING, PAUSED) -> (RESUMING, CANCELING)
    //
    private void updateActiveTask(Task orig, Task mods) {
        // Check for valid state transition
        String oldState = orig.getState();
        String newState = StringUtil.normalize(mods.getState());
        if (newState != null && !newState.equals(oldState)) {
            boolean illegal = false;
            if (oldState.equals(Task.STARTING) || oldState.equals(Task.RUNNING) || oldState.equals(Task.RESUMING)) {
                if (!newState.equals(Task.PAUSING) && !newState.equals(Task.CANCELING)) {
                    illegal = true;
                }
            } else if (oldState.equals(Task.PAUSING) || oldState.equals(Task.PAUSED)) {
                if (!newState.equals(Task.RESUMING) && !newState.equals(Task.CANCELING)) {
                    illegal = true;
                }
            } else {
                illegal = true;
            }
            if (illegal) {
                throw new IllegalArgumentException("Illegal state transition: "
                        + oldState + " -> " + newState);
            }
        }

        // Check that prop changes are not being requested.
        cannotChangeWhileActive("name", orig.getName(), mods.getName());
        cannotChangeWhileActive("type", orig.getType(), mods.getType());
        cannotChangeWhileActive("schedule", orig.getSchedule(), mods.getSchedule());
        cannotChangeWhileActive("data", orig.getData(), mods.getData());

        // Finally, if the state has changed, write it
        if (newState != null && !newState.equals(oldState)) {
            db.update(UPDATE_SQL2, newState, Integer.parseInt(orig.getId()));
        }
    }

    private static void cannotChangeWhileActive(String name, String oldVal, String newVal) {
        if (newVal != null && !newVal.equals(oldVal)) {
            throw new IllegalArgumentException("Cannot change task " + name + " while active");
        }
    }


    public void deleteTask(String id) {
        final int taskId = Integer.parseInt(id);
        tt.execute(new TransactionCallbackWithoutResult() {
            public void doInTransactionWithoutResult(TransactionStatus status) {
                boolean success = false;
                try {
                    db.update("DELETE FROM TaskSetDeps WHERE taskId = ?", taskId);
                    db.update("DELETE FROM TaskStoreDeps WHERE taskId = ?", taskId);
                    db.update("DELETE FROM Tasks WHERE id = ?", taskId);
                    success = true;
                } finally {
                    if (!success) {
                        status.setRollbackOnly();
                    }
                }
            }
        });
    }

}