package org.duraspace.fcrepo.cloudsync.service.dao;

import org.duraspace.fcrepo.cloudsync.api.TaskLog;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Writer;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.List;

public class TaskLogDao extends AbstractDao {

    private static final String CREATE_DDL =
          "CREATE TABLE TaskLogs (\n"
        + "  id INTEGER PRIMARY KEY NOT NULL GENERATED BY DEFAULT AS IDENTITY,\n"
        + "  taskId INTEGER NOT NULL,\n"
        + "  resultType VARCHAR(16) NOT NULL,\n"
        + "  startDate BIGINT NOT NULL,\n"
        + "  finishDate BIGINT,\n"
        + "  CONSTRAINT TaskLog_Task_FK FOREIGN KEY (taskId) REFERENCES Tasks (id))";

    public TaskLogDao(JdbcTemplate db) {
        super(db);
        getTaskLogDir().mkdir();
    }

    @Override
    public void initDb() {
        db.execute(CREATE_DDL);
    }

    // TaskManager use only
    public String start(String taskId) {
        long now = System.currentTimeMillis();
        return insert("INSERT INTO TaskLogs (taskId, resultType, startDate) VALUES (?, ?, ?)",
                Integer.parseInt(taskId), TaskLog.INCOMPLETE, now);
    }

    // TaskManager use only
    public PrintWriter getContentWriter(String id) {
        try {
            OutputStream out = new FileOutputStream(getLogFile(id));
            Writer writer = new OutputStreamWriter(out, "UTF-8");
            return new PrintWriter(writer, true);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    // TaskManager use only
    public Date finish(String id, String resultType) {
        long now = System.currentTimeMillis();
        db.update("UPDATE TaskLogs SET resultType = ?, finishDate = ? WHERE id = ?",
                resultType, now, Integer.parseInt(id));
        return new Date(now);
    }

    public List<TaskLog> listTaskLogs() {
        return db.query("SELECT * FROM TaskLogs ORDER BY finishDate DESC NULLS FIRST",
                new RowMapper<TaskLog>() {
                    public TaskLog mapRow(ResultSet rs, int i) throws SQLException {
                        return getTaskLog(rs);
                    }
                });
    }

    public TaskLog getTaskLog(String id) {
        return db.query("SELECT * FROM TaskLogs WHERE id = ?",
                new ResultSetExtractor<TaskLog>() {
                    public TaskLog extractData(ResultSet rs)
                            throws SQLException {
                        if (rs.next()) {
                            return getTaskLog(rs);
                        } else {
                            return null;
                        }
                    }
                },
                Integer.parseInt(id));
    }

    private static TaskLog getTaskLog(ResultSet rs) throws SQLException {
        TaskLog o = new TaskLog();
        o.setId("" + rs.getInt("id"));
        o.setTaskId("" + rs.getInt("taskId"));
        o.setResultType(rs.getString("resultType"));
        o.setStartDate(getDate(rs, "startDate"));
        o.setFinishDate(getDate(rs, "finishDate"));
        return o;
    }

    private static Date getDate(ResultSet rs, String name) throws SQLException {
        long ms = rs.getLong(name);
        if (ms == 0) {
            return null;
        }
        return new Date(ms);
    }

    public InputStream getTaskLogContent(String id) throws FileNotFoundException {
        return new FileInputStream(getLogFile(id));
    }

    public void deleteTaskLog(String id) {
        db.update("DELETE FROM TaskLogs WHERE id = ?", Integer.parseInt(id));
        getLogFile(id).delete();
    }

    private File getLogFile(String id) {
        Integer.parseInt(id); // ensure the id is numeric
        return new File(getTaskLogDir(), "tasklog-" + id + ".txt");
    }

    private static File getTaskLogDir() {
        File homeDir = new File(System.getProperty("cloudsync.home"));
        return new File(homeDir, "tasklogs");
    }

}