package org.echocat.rundroid.maven.plugins;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.echocat.rundroid.maven.plugins.emulator.EmulatorDaemon;
import org.echocat.rundroid.maven.plugins.emulator.EmulatorMojoSupport;

import javax.annotation.Nonnull;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.maven.plugins.annotations.LifecyclePhase.PRE_INTEGRATION_TEST;
import static org.echocat.jomon.runtime.util.ResourceUtils.closeQuietly;
import static org.echocat.rundroid.maven.plugins.emulator.EmulatorDaemonRequirement.emulatorDaemon;

@Mojo(
    name = "startEmulator",
    defaultPhase = PRE_INTEGRATION_TEST
)
@SuppressWarnings({"InstanceVariableNamingConvention"})
public class StartEmulatorMojo extends EmulatorMojoSupport {

    @Parameter(property = "emulator.arguments")
    private String[] emulatorArguments;
    @Parameter(property = "emulator.output", defaultValue = "redirectToLogger", required = true)
    private String emulatorOutput;
    @Parameter(property = "emulator.logStart", defaultValue = "true")
    private boolean logEmulatorStart;
    @Parameter(property = "emulator.logStartupDone", defaultValue = "true")
    private boolean logEmulatorStartupDone;
    @Parameter(property = "emulator.logTermination", defaultValue = "true")
    private boolean logEmulatorTermination;
    @Parameter(property = "emulator.shutdownWithMaven", defaultValue = "true")
    private boolean shutdownEmulatorWithMaven;

    @Override
    public void call() throws Exception {
        final EmulatorDaemon daemon = (EmulatorDaemon) processDaemonRepository().generate(emulatorDaemon(getTargetEmulatorExecutable(), getAvd())
            .withArguments(getEmulatorArguments())
            .withStreamListener(getEmulatorOutput())
            .whichLogsStartOfProcess(logEmulatorStart)
            .whichLogsStartupDoneOfProcess(logEmulatorStartupDone)
            .whichLogsTerminationOfProcess(logEmulatorTermination)
            .whichShuttingDownWithThisJvm(shutdownEmulatorWithMaven)
            .withAdbExecutable(getTargetAdbExecutable())
        );
        registerPidOf(daemon);
        registerDeviceSerialNumberOf(daemon);
    }

    protected void registerPidOf(@Nonnull EmulatorDaemon daemon) throws MojoExecutionException {
        registerPid(daemon.getProcess(), getEmulatorPidProperty());
    }

    protected void registerDeviceSerialNumberOf(@Nonnull EmulatorDaemon daemon) throws MojoExecutionException {
        final String property = getEmulatorSerialNumberProperty();
        validatePropertyForRegister(daemon, property);
        setProjectProperty(property, daemon.getSerialNumber());
    }

    protected void validatePropertyForRegister(@Nonnull EmulatorDaemon daemon, @Nonnull String property) throws MojoExecutionException {
        final String value = findProjectProperty(property);
        if (!isEmpty(value)) {
            closeQuietly(daemon);
            throw new MojoExecutionException("The project property '" + property + "' is already set to '" + value + "' this indicates double usage of this maven plugin without configuring another pid property.");
        }
    }

    @Nonnull
    protected String[] getEmulatorArguments() throws MojoExecutionException {
        final String[] result = emulatorArguments;
        return result != null && (result.length > 0) ? result : new String[]{"-no-skin", "-no-window"};
    }

    @Nonnull
    protected String getEmulatorOutput() throws MojoExecutionException {
        final String result = emulatorOutput;
        if (result == null) {
            throw new MojoExecutionException("No output provided.");
        }
        return result;
    }

}
