/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins;

import com.android.ddmlib.IDevice;
import com.android.ddmlib.testrunner.IRemoteAndroidTestRunner.TestSize;
import com.android.ddmlib.testrunner.ITestRunListener;
import com.android.ddmlib.testrunner.RemoteAndroidTestRunner;
import com.android.ddmlib.testrunner.TestIdentifier;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.echocat.jomon.runtime.util.Consumer;
import org.echocat.jomon.runtime.util.Duration;
import org.echocat.rundroid.maven.plugins.platform.XmlTestRunListener;
import org.echocat.rundroid.maven.plugins.utils.ManifestAndAdbMojoSupport;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.android.ddmlib.testrunner.ITestRunListener.TestFailure.ERROR;
import static java.lang.String.format;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.apache.commons.io.FileUtils.forceMkdir;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.split;
import static org.apache.maven.plugins.annotations.LifecyclePhase.INTEGRATION_TEST;
import static org.echocat.jomon.runtime.CollectionUtils.asList;
import static org.echocat.rundroid.maven.plugins.utils.ManifestUtils.findInstrumentationRunnerInAndroidManifest;

@SuppressWarnings("InstanceVariableNamingConvention")
@Mojo(
    name = "test",
    defaultPhase = INTEGRATION_TEST
)
public class TestMojo extends ManifestAndAdbMojoSupport {

    @Nonnull
    protected static final String INDENT = "  ";

    @Parameter(readonly = true, defaultValue = "false", property = "maven.test.skip")
    private boolean mavenTestSkip;
    @Parameter(property = "skipTests")
    private Boolean skipTests;
    @Parameter(property = "maven.test.failure.ignore", defaultValue = "false")
    private boolean ignoreTestFailure;
    @Parameter(property = "maven.test.error.ignore", defaultValue = "false")
    private boolean ignoreTestError;

    @Parameter(property = "android.test.timeout", defaultValue = "10m")
    private String testTimeout;

    @Parameter(property = "android.instrumentation.runner")
    private String instrumentationRunner;

    @Parameter(property = "android.test.packages")
    private List<String> testPackages;
    @Parameter(property = "android.test.classes")
    private List<String> testClasses;
    @Parameter(property = "android.test.coverage", defaultValue = "false")
    private boolean testCoverage;
    @Parameter(property = "android.test.debug", defaultValue = "false")
    private boolean testDebug;
    @Parameter(property = "android.test.logOnly", defaultValue = "false")
    private boolean testLogOnly;
    @Parameter(property = "android.test.size")
    private TestSize testSize;
    @Parameter(property = "android.test.reports.directory", defaultValue = "${project.build.directory}/surefire-reports")
    private File reportsDirectory;

    @Override
    public void call() throws Exception {
        if (isEnableIntegrationTest()) {
            doWithDevices(new Consumer<IDevice, Exception>() { @Override public void consume(@Nullable IDevice device) throws Exception {
                for (final String testPackage : getTestPackages()) {
                    final RemoteAndroidTestRunner runner = new RemoteAndroidTestRunner(getInstrumentationPackage(), getInstrumentationRunner(), device);
                    if (!testPackage.isEmpty()) {
                        runner.setTestPackageName(testPackage);
                    }
                    final String[] classes = getTestClassesAsArray();
                    if (classes.length > 0) {
                        runner.setClassNames(classes);
                    }
                    runner.setCoverage(testCoverage);
                    runner.setDebug(testDebug);
                    runner.setLogOnly(testLogOnly);
                    if (testSize != null) {
                        runner.setTestSize(testSize);
                    }
                    runner.setMaxtimeToOutputResponse((int) getTestTimeout().in(MILLISECONDS));
                    final TestRunListener listener = createTestRunListener();
                    runner.run(createJunitReportListener(device), listener);
                    if (listener.getErrorMessage() != null) {
                        throw new MojoFailureException(listener.getErrorMessage());
                    }
                }
            }});
        } else {
            getLog().info("Skipping tests");
        }
    }

    @Nonnull
    protected XmlTestRunListener createJunitReportListener(@Nonnull IDevice device) throws MojoExecutionException, IOException {
        final File reportDir = getReportsDirectory();
        forceMkdir(reportDir);
        return new XmlTestRunListener(device, reportDir);
    }

    @Nonnull
    protected TestRunListener createTestRunListener() throws MojoExecutionException {
        return new TestRunListener();
    }

    protected boolean isEnableIntegrationTest() throws MojoFailureException, MojoExecutionException {
        final boolean skip;
        if (skipTests != null) {
            skip = skipTests;
        } else {
            skip = mavenTestSkip;
        }
        return !skip;
    }

    @Nonnull
    protected File getReportsDirectory() throws MojoExecutionException {
        return get(reportsDirectory, "reportsDirectory");
    }

    @Nonnull
    protected List<String> getTestPackages() throws MojoExecutionException {
        final List<String> result = splitUp(testPackages);
        return result.isEmpty() ? asList("") : result;
    }

    @Nonnull
    protected List<String> getTestClasses() throws MojoExecutionException {
        return splitUp(testClasses);
    }

    @Nonnull
    protected String[] getTestClassesAsArray() throws MojoExecutionException {
        final List<String> classes = getTestClasses();
        return classes.toArray(new String[classes.size()]);
    }

    @Nonnull
    protected Duration getTestTimeout() throws MojoExecutionException {
        return getDuration(testTimeout, "testTimeout");
    }

    @Nullable
    protected String getInstrumentationRunner() throws MojoExecutionException {
        String result = instrumentationRunner;
        if (isEmpty(result)) {
            final File file = findManifestFile();
            if (file != null) {
                result = findInstrumentationRunnerInAndroidManifest(file);
            }
        }
        return result;
    }

    @Nonnull
    protected List<String> splitUp(@Nullable List<String> inputs) throws MojoExecutionException {
        final List<String> result = new ArrayList<>();
        if (inputs != null) {
            for (final String plainInput : inputs) {
                for (final String subPlainInput : split(plainInput, ',')) {
                    final String trimmedInput = subPlainInput.trim();
                    if (!trimmedInput.isEmpty()) {
                        result.add(trimmedInput);
                    }
                }
            }
        }
        return result;
    }

    protected class TestRunListener implements ITestRunListener {

        @Nonnegative
        private int _testCount;
        @Nonnegative
        private int _testRunCount;
        @Nonnegative
        private int _testErrorCount;
        @Nonnegative
        private int _testFailureCount;

        @Nullable
        private String _errorMessage;

        @Override
        public void testRunStarted(String runName, int testCount) {
            _testCount = testCount;
            getLog().info(INDENT + "Run started: " + runName + ", " + testCount + " tests:" );
        }

        @Override
        public void testStarted(TestIdentifier identifier) {
            _testRunCount++;
            getLog().info(format("%1$s%1$sStart [%2$d/%3$d]: %4$s", INDENT, _testRunCount, _testCount, identifier.toString()));
        }

        @Override
        public void testFailed(TestFailure status, TestIdentifier identifier, String trace) {
            if (status == ERROR) {
                _testErrorCount++;
            } else {
                _testFailureCount++;
            }
            getLog().error(INDENT + INDENT + status.name() + ":" + identifier);
            getLog().error(INDENT + INDENT + trace);
        }

        @Override
        public void testEnded(TestIdentifier identifier, Map<String, String> testMetrics) {
            getLog().info(format("%1$s%1$sEnd [%2$d/%3$d]: %4$s", INDENT, _testRunCount, _testCount, identifier.toString()));
            logMetrics( testMetrics );
        }

        @Override
        public void testRunFailed(String errorMessage) {
            _errorMessage = errorMessage;
            getLog().info(INDENT + "Run failed: " + errorMessage);
        }

        @Override
        public void testRunStopped(long elapsedTime) {
            getLog().info(INDENT + "Run stopped:" + new Duration(elapsedTime));
        }

        @Override
        public void testRunEnded(long elapsedTime, Map<String, String> runMetrics) {
            getLog().info(INDENT + "Run ended: " + new Duration(elapsedTime));
            final String message = INDENT + "Tests run: " + _testRunCount + (_testRunCount < _testCount ? " (of " + _testCount + ")" : "") + ",  Failures: " + _testFailureCount + ",  Errors: " + _testErrorCount;
            if (hasFailuresOrErrors()) {
                getLog().error(message);
            } else {
                getLog().info(message);
            }
        }

        protected void logMetrics(@Nonnull Map<String, String> metrics) {
            for (final Map.Entry<String, String> entry : metrics.entrySet()) {
                getLog().info(INDENT + INDENT + entry.getKey() + ": " + entry.getValue());
            }
        }

        protected boolean hasFailuresOrErrors() {
            final boolean result;
            if (ignoreTestFailure && ignoreTestError) {
                result = false;
            } else if (ignoreTestFailure) {
                result = _testErrorCount > 0;
            } else if (ignoreTestError) {
                result = _testFailureCount > 0;
            } else {
                result = _testErrorCount > 0 || _testFailureCount > 0;
            }
            return result;
        }

        @Nullable
        protected String getErrorMessage() {
            return _errorMessage;
        }

    }
}
