/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.emulator;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Parameter;
import org.echocat.rundroid.maven.plugins.utils.AndroidMojoSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.echocat.jomon.process.ExecutableDiscovery.Task.executable;
import static org.echocat.jomon.process.ExecutableDiscovery.executableDiscovery;

@SuppressWarnings("InstanceVariableNamingConvention")
public abstract class EmulatorMojoSupport extends AndroidMojoSupport {

    @Parameter(property = "emulator.executable")
    private String emulatorExecutable;
    @Parameter(property = "emulator.architecture", defaultValue = "x86", required = true)
    private String emulatorArchitecture;
    @Parameter(property = "emulator.avd", defaultValue = "default", required = true)
    private String avd;
    @Parameter(property = "emulator.pidProperty", defaultValue = "emulator.pid", required = true)
    private String emulatorPidProperty;
    @Parameter(property = "emulator.emulatorSerialNumberProperty", defaultValue = "android.device.serialNumber", required = true)
    private String emulatorSerialNumberProperty;

    @Nullable
    protected String getEmulatorExecutable() throws MojoExecutionException {
        return emulatorExecutable;
    }

    @Nonnull
    protected File getTargetEmulatorExecutable() throws MojoExecutionException {
        final String executableFileName = getTargetEmulatorExecutableFileName();
        final File executable = executableDiscovery().discover(executable(executableFileName)
            .withinEnvironmentVariables("ANDROID_HOME")
            .searchInSubDirectories("tools")
        );
        if (executable == null) {
            throw new IllegalArgumentException("Could not find emulator executable '" + executableFileName + "' neither as absolute path nor in environment variable ANDROID_HOME nor in system PATH.");
        }
        return executable;
    }

    @Nonnull
    protected String getTargetEmulatorExecutableFileName() throws MojoExecutionException {
        final String executableFileNameFromRequirement = getEmulatorExecutable();
        final String executableFileName;
        if (isEmpty(executableFileNameFromRequirement)) {
            executableFileName = "emulator-" + getEmulatorArchitecture();
        } else {
            executableFileName = executableFileNameFromRequirement;
        }
        return executableFileName;
    }

    @Nonnull
    protected String getEmulatorArchitecture() throws MojoExecutionException {
        final String result = emulatorArchitecture;
        if (result == null) {
            throw new MojoExecutionException("No architecture set.");
        }
        return result;
    }

    @Nonnull
    protected String getAvd() throws MojoExecutionException {
        final String result = avd;
        if (result == null) {
            throw new MojoExecutionException("No avd set.");
        }
        return result;
    }

    @Nonnull
    protected String getEmulatorPidProperty() throws MojoExecutionException {
        final String result = emulatorPidProperty;
        if (result == null) {
            throw new MojoExecutionException("No emulatorPidProperty set.");
        }
        return result;
    }

    @Nonnull
    public String getEmulatorSerialNumberProperty() throws MojoExecutionException {
        final String result = emulatorSerialNumberProperty;
        if (result == null) {
            throw new MojoExecutionException("No emulatorSerialNumberProperty set.");
        }
        return result;
    }

}
