/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.platform;

import com.android.ddmlib.*;
import org.echocat.jomon.process.GeneratedProcess;
import org.echocat.jomon.process.listeners.stream.StreamListener;
import org.echocat.jomon.runtime.util.Consumer;
import org.echocat.jomon.runtime.util.Duration;
import org.echocat.rundroid.maven.plugins.platform.DeviceController.Environment;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import static java.lang.Thread.currentThread;
import static org.echocat.jomon.process.ProcessUtils.toEscapedCommandLine;
import static org.echocat.jomon.runtime.CollectionUtils.asImmutableList;
import static org.echocat.jomon.runtime.io.StreamType.stdout;
import static org.echocat.rundroid.maven.plugins.platform.DeviceController.Environment.deviceEnvironment;
import static org.echocat.rundroid.maven.plugins.platform.DeviceController.deviceController;

public class AdbProcess implements GeneratedProcess<String, Long> {

    @Nonnegative
    private final long _id;
    @Nonnull
    private final String _executable;
    @Nonnull
    private final List<String> _arguments;
    @Nonnull
    private final IDevice _device;
    @Nullable
    private final Duration _timeout;
    @Nonnull
    private final StreamListener<AdbProcess> _listener;
    @Nullable
    private final Consumer<Exception, RuntimeException> _callOnDone;

    @Nullable
    private volatile Thread _executionThread;
    @Nullable
    private volatile Exception _executionException;

    private volatile boolean _killed;

    public AdbProcess(@Nonnegative long id, @Nonnull String executable, @Nonnull Iterable<String> arguments, @Nonnull IDevice device, @Nullable Duration timeout, @Nonnull StreamListener<AdbProcess> listener) {
        this(id, executable, arguments, device, timeout, listener, null);
    }

    public AdbProcess(@Nonnegative long id, @Nonnull String executable, @Nonnull Iterable<String> arguments, @Nonnull IDevice device, @Nullable Duration timeout, @Nonnull StreamListener<AdbProcess> listener, @Nullable Consumer<Exception, RuntimeException> callOnDone) {
        _id = id;
        _executable = executable;
        _device = device;
        _timeout = timeout;
        _listener = listener;
        _callOnDone = callOnDone;
        _arguments = asImmutableList(arguments);
        _executionThread = start();
    }

    @Nonnull
    protected Thread start() {
        final Environment environment = deviceEnvironment(_device)
            .withCommandTimeout(_timeout)
            .withProgressConsumer(getOutputStreamOnListener());

        final Thread thread = new Thread("Wait for " + _executable + " on " + _device.getSerialNumber()) { @Override public void run() {
            try {
                deviceController().exec(environment, toEscapedCommandLine(_executable, _arguments), new Receiver());
            } catch (TimeoutException | IOException | ShellCommandUnresponsiveException | AdbCommandRejectedException e) {
                _executionException = e;
            } finally {
                _executionThread = null;
                if (_callOnDone != null) {
                    _callOnDone.consume(_executionException);
                }
            }
        }};
        thread.start();
        return thread;
    }

    @Override
    public int waitFor() throws InterruptedException {
        if (_executionThread != null) {
            _executionThread.join();
        }
        return exitValue();
    }

    @Override
    public int exitValue() throws IllegalStateException {
        if (isAlive()) {
            throw new IllegalStateException("The process is still running.");
        }
        return _executionException != null ? 1 : 0;
    }

    @Override
    public boolean isAlive() {
        return _executionThread != null && _executionThread.isAlive();
    }

    @Override
    public boolean isDaemon() {
        return false;
    }

    @Override
    public void kill() throws IOException {
        close();
    }

    @Override
    public void close() throws IOException {
        _killed = true;
        try {
            waitFor();
        } catch (final InterruptedException ignored) {
            currentThread().interrupt();
        }
    }

    @Nullable
    @Override
    public String getExecutable() {
        return _executable;
    }

    @Nullable
    @Override
    public List<String> getArguments() {
        return _arguments;
    }

    @Nullable
    @Override
    public Long getId() {
        return _id;
    }

    @Nonnull
    @Override
    public OutputStream getStdin() throws IOException {
        throw new UnsupportedOperationException();
    }

    @Nonnull
    @Override
    public InputStream getStdout() throws IOException {
        throw new UnsupportedOperationException();
    }

    @Nonnull
    @Override
    public InputStream getStderr() throws IOException {
        throw new UnsupportedOperationException();
    }

    @Nonnull
    protected OutputStream getOutputStreamOnListener() {
        return new OutputStream() {

            @Override
            public void write(int b) throws IOException {
                write(new byte[]{(byte) b}, 0, 1);
            }

            @Override
            public void write(byte[] b, int off, int len) throws IOException {
                _listener.notifyOutput(AdbProcess.this, b, len, len, stdout);
            }

            @Override
            public void flush() throws IOException {
                _listener.flushOutput(AdbProcess.this, stdout);
            }

        };
    }

    protected class Receiver implements IShellOutputReceiver {

        @Override
        public void addOutput(byte[] data, int offset, int length) {
            _listener.notifyOutput(AdbProcess.this, data, offset, length, stdout);
        }

        @Override
        public void flush() {
            _listener.flushOutput(AdbProcess.this, stdout);
        }

        @Override
        public boolean isCancelled() {
            return _killed;
        }

    }

}
