/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.platform;

import com.android.ddmlib.*;
import com.android.ddmlib.SyncService.ISyncProgressMonitor;
import org.echocat.jomon.runtime.util.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.WillNotClose;
import java.io.*;
import java.text.MessageFormat;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.android.ddmlib.SyncException.SyncError.CANCELED;
import static java.lang.System.out;
import static java.util.Locale.US;
import static java.util.UUID.randomUUID;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.apache.commons.io.FilenameUtils.getExtension;
import static org.apache.commons.lang3.StringUtils.isEmpty;

public class DeviceController {

    @Nonnull
    private static final Logger LOG = LoggerFactory.getLogger(DeviceController.class);
    @Nonnull
    private static final DeviceController INSTANCE = new DeviceController();

    @Nonnull
    public static DeviceController getInstance() {
        return INSTANCE;
    }

    @Nonnull
    public static DeviceController deviceController() {
        return getInstance();
    }

    public void transferLocalFile(@Nonnull Environment environment, @Nonnull File localFile, @Nonnull String remoteFile) throws IOException, TimeoutException, SyncException {
        try {
            final IDevice device = environment.getDevice();
            final SyncService sync = device.getSyncService();
            if (sync != null) {
                try {
                    sync.pushFile(localFile.getPath(), remoteFile, getSyncProgressMonitor(environment, localFile.getPath()));
                } finally {
                    sync.close();
                }
            } else {
                throw new SyncException(CANCELED, "Unable to open sync connection.");
            }
        } catch (final AdbCommandRejectedException e) {
            throw new SyncException(CANCELED, e);
        }
    }

    public void installLocalFile(@Nonnull Environment environment, @Nonnull File localFile, @Nonnull String packageName, boolean reinstall) throws IOException, SyncException, InstallException, TimeoutException {
        final String remoteFile = "/data/local/tmp/" + randomUUID() + "." + getExtension(localFile.getName());
        try {
            transferLocalFile(environment, localFile, remoteFile);
            installRemoteFile(environment, remoteFile, localFile.getName(), packageName, reinstall);
        } finally {
            deleteRemoveFile(environment, remoteFile);
        }
    }

    public void installRemoteFile(@Nonnull Environment environment, @Nonnull String remoteFile, @Nonnull String packageName, boolean reinstall) throws InstallException, TimeoutException, IOException {
        installRemoteFile(environment, remoteFile, remoteFile, packageName, reinstall);
    }

    protected void installRemoteFile(@Nonnull Environment environment, @Nonnull String remoteFile, @Nonnull String fileDisplay, @Nonnull String packageName, boolean reinstall) throws InstallException, TimeoutException, IOException {
        final AtomicInteger writtenToConsole = new AtomicInteger();
        final IDevice device = environment.getDevice();
        final InstallReceiver receiver = getInstallReceiver(environment);
        final String options = reinstall ? "-r" : "";
        write(environment, writtenToConsole, "\rInstalling " + fileDisplay + " on " + device.getSerialNumber() + "...");
        try {
            try {
                if (reinstall) {
                    exec(environment, "pm uninstall \"" + packageName + "\"", NullOutputReceiver.getReceiver());
                }
                exec(environment, "pm install " + options + " \"" + remoteFile + "\"", receiver);
            } catch (AdbCommandRejectedException | ShellCommandUnresponsiveException e) {
                throw new InstallException("Could not install " + remoteFile + " to " + device.getSerialNumber() + ". Got: " + e.getCause(), e);
            }
            final String errorMessage = receiver.getErrorMessage();
            if (!isEmpty(errorMessage)) {
                throw new InstallException("Could not install " + remoteFile + " to " + device.getSerialNumber() + ". Got: " + errorMessage, null);
            }
        } finally {
            cleanupConsoleIfNeeded(environment, writtenToConsole);
        }
        LOG.info("Installed " + fileDisplay + " on " + device.getSerialNumber() + ".");
    }

    public void exec(@Nonnull Environment environment, @Nonnull String command, @Nullable IShellOutputReceiver receiver) throws TimeoutException, AdbCommandRejectedException, ShellCommandUnresponsiveException, IOException {
        final IDevice device = environment.getDevice();
        final Duration timeout = environment.getCommandTimeout();
        device.executeShellCommand(command, receiver != null ? receiver : getDefaultShellOutputReceiver(environment), timeout != null ? (int) timeout.in(MILLISECONDS) : 0);
    }

    public void exec(@Nonnull Environment environment, @Nonnull String command) throws TimeoutException, AdbCommandRejectedException, ShellCommandUnresponsiveException, IOException {
        exec(environment, command, null);
    }

    public void deleteRemoveFile(@Nonnull Environment environment, @Nonnull String remoteFilePath) throws SyncException, TimeoutException, IOException {
        try {
            exec(environment, "rm \"" + remoteFilePath + "\"");
        } catch (AdbCommandRejectedException | ShellCommandUnresponsiveException e) {
            throw new SyncException(CANCELED, e);
        }
    }

    protected static void write(@Nonnull Environment environment, @Nonnull AtomicInteger writtenToConsole, @Nonnull String message) throws IOException {
        final Writer out = environment.getProgressConsumer();
        if (out != null && LOG.isInfoEnabled()) {
            out.write('\r');
            out.write(message);
            out.flush();
            final int diff = writtenToConsole.get() - message.length();
            for (int i = 0; i < diff; i++) {
                out.write(' ');
            }
            writtenToConsole.set(message.length());
        }
    }

    protected static void cleanupConsoleIfNeeded(@Nonnull Environment environment, @Nonnull AtomicInteger writtenToConsole) throws IOException {
        write(environment, writtenToConsole, "");
        final Writer out = environment.getProgressConsumer();
        if (out != null && LOG.isInfoEnabled()) {
            out.write('\r');
        }
    }

    @Nonnull
    protected SyncProgressMonitorImpl getSyncProgressMonitor(@Nonnull Environment environment, @Nonnull String from) {
        return new SyncProgressMonitorImpl(environment, from);
    }

    @Nonnull
    protected IShellOutputReceiver getDefaultShellOutputReceiver(@SuppressWarnings("UnusedParameters") @Nonnull Environment environment) {
        return new RedirectToLogShellOutputReceiver();
    }

    @Nonnull
    protected InstallReceiver getInstallReceiver(@SuppressWarnings("UnusedParameters") @Nonnull Environment environment) {
        return new InstallReceiver();
    }

    @Nonnull
    protected static class RedirectToLogShellOutputReceiver extends MultiLineReceiver {

        @Override
        public void processNewLines(String[] lines) {
            if (lines != null) {
                for (final String line : lines) {
                    LOG.info(line);
                }
            }
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

    }

    protected static class SyncProgressMonitorImpl implements ISyncProgressMonitor {

        @Nonnull
        protected static final MessageFormat PROGRESS_START_FORMAT = new MessageFormat("Transferring {0} to {1}...", US);
        @Nonnull
        protected static final MessageFormat PROGRESS_FORMAT = new MessageFormat("Transferring {0} to {1}... {2,number,#.0}%", US);
        @Nonnull
        protected static final MessageFormat PROGRESS_DONE_FORMAT = new MessageFormat("Transferred {0} to {1}.", US);

        @Nonnull
        private final AtomicInteger _writtenToConsole = new AtomicInteger();
        @Nonnull
        private final Environment _environment;
        @Nonnull
        private final String _deviceString;
        @Nonnull
        private final String _from;

        @Nonnegative
        private int _totalWork;
        @Nonnegative
        private int _currentWork;

        public SyncProgressMonitorImpl(@Nonnull Environment environment, @Nonnull String from) {
            _environment = environment;
            _deviceString = getDeviceStringFor(environment.getDevice());
            _from = from;
        }

        @Override
        public void start(int totalWork) {
            _totalWork = totalWork;
            final String message = PROGRESS_START_FORMAT.format(new Object[]{_from, _deviceString});
            try {
                write(_environment, _writtenToConsole, message);
            } catch (final IOException e) {
                LOG.warn("Could not log output: " + message, e);
            }
        }

        @Override
        public void stop() {
            try {
                cleanupConsoleIfNeeded(_environment, _writtenToConsole);
            } catch (final IOException e) {
                LOG.warn("Could not clear output.", e);
            }
            LOG.info(PROGRESS_DONE_FORMAT.format(new Object[]{_from, _deviceString}));
        }

        @Override
        public void advance(@Nonnegative int work) {
            _currentWork += work;
            final float plainProgress = ((float)_currentWork * 100f) / (float) _totalWork;
            final float progress = plainProgress < 100f ? plainProgress : 100f;
            final String message = PROGRESS_FORMAT.format(new Object[]{_from, _deviceString, progress});
            try {
                write(_environment, _writtenToConsole, message);
            } catch (final IOException e) {
                LOG.debug("Could not log output: " + message, e);
            }
        }

        @Nonnull
        protected String getDeviceStringFor(@Nonnull IDevice device) {
            return device.getSerialNumber();
        }

        @Nonnegative
        protected int getTotalWork() {
            return _totalWork;
        }

        @Override
        public void startSubTask(String name) {}

        @Override
        public boolean isCanceled() { return false; }
    }

    protected static class InstallReceiver extends MultiLineReceiver {

        @Nonnull
        protected static final String SUCCESS_OUTPUT = "Success";
        @Nonnull
        protected static final Pattern FAILURE_PATTERN = Pattern.compile("Failure\\s+\\[(.*)\\]");

        @Nullable
        private String _errorMessage;

        @Override
        public void processNewLines(String[] lines) {
            for (final String line : lines) {
                if (!line.isEmpty()) {
                    if (line.startsWith(SUCCESS_OUTPUT)) {
                        _errorMessage = null;
                    } else {
                        final Matcher matcher = FAILURE_PATTERN.matcher(line);
                        if (matcher.matches()) {
                            _errorMessage = matcher.group(1);
                        }
                    }
                }
            }
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

        @Nullable
        public String getErrorMessage() {
            return _errorMessage;
        }
    }

    public static class Environment {

        @Nonnull
        public static Environment deviceEnvironment(@Nonnull IDevice device) {
            return new Environment(device);
        }

        @Nonnull
        private final IDevice _device;

        @SuppressWarnings("UseOfSystemOutOrSystemErr")
        @Nullable
        private Writer _progressConsumer = new OutputStreamWriter(out);
        @Nullable
        private Duration _commandTimeout = new Duration("2m");

        public Environment(@Nonnull IDevice device) {
            _device = device;
        }

        @Nonnull
        public Environment withCommandTimeout(@Nullable Duration timeout) {
            _commandTimeout = timeout;
            return this;
        }

        @Nonnull
        public Environment withCommandTimeout(@Nullable String timeout) {
            return withCommandTimeout(timeout != null ? new Duration(timeout) : null);
        }

        @Nonnull
        public Environment withoutCommandTimeout() {
            return withCommandTimeout((Duration) null);
        }

        @Nonnull
        public Environment withProgressConsumer(@Nullable @WillNotClose OutputStream os) {
            return withProgressConsumer(os != null ? new OutputStreamWriter(os) : null);
        }

        @Nonnull
        public Environment withProgressConsumer(@Nullable @WillNotClose Writer writer) {
            _progressConsumer = writer;
            return this;
        }

        @Nonnull
        public IDevice getDevice() {
            return _device;
        }

        @Nullable
        public Duration getCommandTimeout() {
            return _commandTimeout;
        }

        @Nullable
        @WillNotClose
        public Writer getProgressConsumer() {
            return _progressConsumer;
        }

    }

}
