/*****************************************************************************************
 * *** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 2.0
 *
 * echocat Maven Rundroid Plugin, Copyright (c) 2012-2013 echocat
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * *** END LICENSE BLOCK *****
 ****************************************************************************************/

package org.echocat.rundroid.maven.plugins.platform;

import com.android.ddmlib.IDevice;
import com.android.ddmlib.testrunner.ITestRunListener;
import com.android.ddmlib.testrunner.TestIdentifier;
import com.android.ddmlib.testrunner.TestResult;
import com.android.ddmlib.testrunner.TestResult.TestStatus;
import org.echocat.rundroid.maven.plugins.utils.DeviceUtils;
import org.kxml2.io.KXmlSerializer;

import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.Map.Entry;

import static com.android.ddmlib.testrunner.TestResult.TestStatus.PASSED;
import static org.echocat.rundroid.maven.plugins.utils.IdentifierUtils.normalize;

public class XmlTestRunListener implements ITestRunListener {

    private static final String TEST_RESULT_FILE_SUFFIX = ".xml";
    private static final String TEST_RESULT_FILE_PREFIX = "TEST-";

    private static final String TESTSUITE = "testsuite";
    private static final String TESTCASE = "testcase";
    private static final String ERROR = "error";
    private static final String FAILURE = "failure";
    private static final String ATTR_NAME = "name";
    private static final String ATTR_VALUE = "value";
    private static final String ATTR_TIME = "time";
    private static final String ATTR_ERRORS = "errors";
    private static final String ATTR_FAILURES = "failures";
    private static final String ATTR_TESTS = "tests";
    private static final String PROPERTIES = "properties";
    private static final String PROPERTY = "property";
    private static final String ATTR_CLASSNAME = "classname";
    private static final String TIMESTAMP = "timestamp";
    private static final String HOSTNAME = "hostname";

    private static final String NS = null;

    @Nonnull
    private final IDevice _device;
    @Nonnull
    private final File _reportDirectory;

    @Nonnull
    private TestRunResult _testRunResult = new TestRunResult();

    public XmlTestRunListener(@Nonnull IDevice device, @Nonnull File reportDirectory) {
        _device = device;
        _reportDirectory = reportDirectory;
    }

    @Override
    public void testRunStarted(String runName, int numTests) {
        _testRunResult = new TestRunResult(runName);
    }

    @Override
    public void testStarted(TestIdentifier test) {
        _testRunResult.reportTestStarted(test);
    }

    @Override
    public void testFailed(TestFailure status, TestIdentifier test, String trace) {
        if (status.equals(TestFailure.ERROR)) {
            _testRunResult.reportTestFailure(test, TestStatus.ERROR, trace);
        } else {
            _testRunResult.reportTestFailure(test, TestStatus.FAILURE, trace);
        }
    }

    @Override
    public void testEnded(TestIdentifier test, Map<String, String> testMetrics) {
        _testRunResult.reportTestEnded(test, testMetrics);
    }

    @Override
    public void testRunFailed(String errorMessage) {
        _testRunResult.setRunFailureError(errorMessage);
    }

    @Override
    public void testRunStopped(long arg0) {
        // ignore
    }

    @Override
    public void testRunEnded(long elapsedTime, Map<String, String> runMetrics) {
        _testRunResult.setRunComplete(true);
        try {
            generateDocument(elapsedTime, _testRunResult, _reportDirectory);
        } catch (final IOException e) {
            throw new RuntimeException("Could not write document. Got: " + e.getMessage(), e);
        }
    }

    private void generateDocument(@Nonnegative long elapsedTime, @Nonnull TestRunResult testRunResult, @Nonnull File reportDirectory) throws IOException {
        try (final OutputStream stream = createOutputResultStream(reportDirectory, testRunResult)) {
            final KXmlSerializer serializer = new KXmlSerializer();
            serializer.setOutput(stream, "UTF-8");
            serializer.startDocument("UTF-8", null);
            serializer.setFeature("http://xmlpull.org/v1/doc/features.html#indent-output", true);
            printTestResults(serializer, elapsedTime, testRunResult);
            serializer.endDocument();
        }
    }

    @Nonnull
    protected String getCurrentTimestamp() {
        final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.getDefault());
        final TimeZone gmt = TimeZone.getTimeZone("UTC");
        dateFormat.setTimeZone(gmt);
        dateFormat.setLenient(true);
        final String timestamp = dateFormat.format(new Date());
        return timestamp;
    }

    @Nonnull
    protected File getResultFile(@Nonnull File reportDir, @Nonnull TestRunResult testRunResult) throws IOException {
        return new File(reportDir, TEST_RESULT_FILE_PREFIX + normalize(testRunResult.getName()) + TEST_RESULT_FILE_SUFFIX);
    }

    @Nonnull
    protected OutputStream createOutputResultStream(@Nonnull File reportDir, @Nonnull TestRunResult testRunResult) throws IOException {
        final File reportFile = getResultFile(reportDir, testRunResult);
        return new FileOutputStream(reportFile);
    }

    protected void printTestResults(@Nonnull KXmlSerializer serializer, @Nonnegative long elapsedTime, @Nonnull TestRunResult testRunResult) throws IOException {
        serializer.startTag(NS, TESTSUITE);
        final String name = testRunResult.getName();
        if (name != null) {
            serializer.attribute(NS, ATTR_NAME, name);
        }
        serializer.attribute(NS, ATTR_TESTS, Integer.toString(testRunResult.getNumTests()));
        serializer.attribute(NS, ATTR_FAILURES, Integer.toString(testRunResult.getNumberOfFailedTests()));
        serializer.attribute(NS, ATTR_ERRORS, Integer.toString(testRunResult.getNumberOfErrorTests()));
        serializer.attribute(NS, ATTR_TIME, Double.toString((double) elapsedTime / 1000.f));
        serializer.attribute(NS, TIMESTAMP, getCurrentTimestamp());
        serializer.attribute(NS, HOSTNAME, DeviceUtils.toString(_device));
        serializer.startTag(NS, PROPERTIES);
        for (final Entry<String, String> entry : new TreeMap<>(_device.getProperties()).entrySet()) {
            serializer.startTag(NS, PROPERTY);
            serializer.attribute(NS, ATTR_NAME, entry.getKey());
            serializer.attribute(NS, ATTR_VALUE, entry.getValue());
            serializer.endTag(NS, PROPERTY);
        }
        serializer.endTag(NS, PROPERTIES);
        final Map<TestIdentifier, TestResult> testResults = testRunResult.getTestResults();
        for (final Map.Entry<TestIdentifier, TestResult> testEntry : testResults.entrySet()) {
            print(serializer, testEntry.getKey(), testEntry.getValue());
        }
        serializer.endTag(NS, TESTSUITE);
    }

    protected void print(@Nonnull KXmlSerializer serializer, @Nonnull TestIdentifier testId, @Nonnull TestResult testResult) throws IOException {
        serializer.startTag(NS, TESTCASE);
        serializer.attribute(NS, ATTR_NAME, testId.getTestName());
        serializer.attribute(NS, ATTR_CLASSNAME, testId.getClassName());
        final long elapsedTimeMs = testResult.getEndTime() - testResult.getStartTime();
        serializer.attribute(NS, ATTR_TIME, Double.toString((double) elapsedTimeMs / 1000.f));
        if (!PASSED.equals(testResult.getStatus())) {
            final String result = testResult.getStatus().equals(TestStatus.FAILURE) ? FAILURE : ERROR;
            serializer.startTag(NS, result);
            final String stackText = sanitize(testResult.getStackTrace());
            serializer.text(stackText);
            serializer.endTag(NS, result);
        }
        serializer.endTag(NS, TESTCASE);
    }

    @Nonnull
    protected String sanitize(@Nonnull String text) {
        return text.replace("\0", "<\\0>");
    }
}
